use super::Iconize;
use crate::Icon;
use bytesize::ByteSize;
use std::fmt;

pub struct NetworkFormatter {
    logical_address: String,
    rx_bytes: usize,
    tx_bytes: usize,
}

impl NetworkFormatter {
    pub fn new() -> Self {
        Self {
            logical_address: String::from("N/A"),
            rx_bytes: 0,
            tx_bytes: 0,
        }
    }

    pub fn set_logical_address(&mut self, addr: String) {
        self.logical_address = addr
    }

    pub fn get_logical_address(&self) -> &str {
        &self.logical_address
    }

    pub fn set_rx_bytes(&mut self, bytes: usize) {
        self.rx_bytes = bytes;
    }

    pub fn set_tx_bytes(&mut self, bytes: usize) {
        self.tx_bytes = bytes;
    }

    pub fn get_rx_bytes(&self) -> ByteSize {
        ByteSize::kb(self.rx_bytes as u64 / 1024)
    }

    pub fn get_tx_bytes(&self) -> ByteSize {
        ByteSize::kb(self.tx_bytes as u64 / 1024)
    }
}

impl Iconize for NetworkFormatter {
    fn set_icon(&self, icon: &mut Icon) {
        icon.find_icon("network-wireless-symbolic")
    }
}

impl fmt::Display for NetworkFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if self.rx_bytes == 0 && self.tx_bytes == 0 {
            return write!(f, "Local IP: {}", self.logical_address);
        }

        write!(
            f,
            "Local IP: {}\nData: {} 🠗 / {} 🠕",
            self.get_logical_address(),
            self.get_rx_bytes(),
            self.get_tx_bytes(),
        )
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display_network() {
        let mut fmt = NetworkFormatter::new();

        fmt.set_logical_address("127.0.0.1".to_owned());
        assert_eq!(fmt.to_string(), "Local IP: 127.0.0.1".to_owned());

        fmt.set_logical_address("192.168.1.1".to_owned());
        fmt.set_rx_bytes(717000);
        fmt.set_tx_bytes(717000);
        assert_eq!(
            fmt.to_string(),
            "Local IP: 192.168.1.1\nData: 700.0 KB 🠗 / 700.0 KB 🠕".to_owned()
        );
    }
}
