#![allow(unused_must_use)]

// The flow of citron is as follows:
//
//   read   ---->   format   ---->   notify
//    |              |                |
//    |              |                |
//    |              |                `---> `notifier` module
//    |              |
//    |              `---> `fmt` modules
//    |
//    `--> `read` module

mod debug;
mod fmt;
mod helpers;
mod icon;
mod notifier;
mod options;
mod read;

use debug::get_debug;
use icon::Icon;
use notifier::{Notifier, Summary};
use options::Options;
use owo_colors::OwoColorize;
use std::env;
use structopt::StructOpt;

fn main() {
    let mut notifier = Notifier::new();
    let mut icon = Icon::new();
    let opt = Options::from_args();

    if env::args().count() == 1 {
        get_started();
        return;
    }

    if opt.debug {
        get_debug();
        return;
    }

    if opt.version {
        get_version();
        return;
    }

    if opt.list_icon_themes {
        println!("Available icon themes:");
        icon.get_available_themes()
            .into_iter()
            .filter(|x| !x.eq("default"))
            .for_each(|x| eprintln!(" - {}", x.yellow()));
    }

    if opt.icons {
        if let Some(theme) = opt.icon_theme.as_ref() {
            icon.make_visible();
            icon.set_theme(theme.to_string());
        }
    }

    if let Some(timeout) = opt.timeout {
        notifier.set_timeout(timeout);
    }

    if opt.uptime {
        notifier.set_summary(Summary::Uptime);
        notifier.set_body(read::uptime(&mut icon));
        notifier.set_icon(&icon);
        notifier.send();
        return;
    }

    if opt.date {
        notifier.set_summary(Summary::Date);
        notifier.set_body(read::date(opt.railway, &mut icon));
        notifier.set_icon(&icon);
        notifier.send();
        return;
    }

    if opt.backlight {
        notifier.set_summary(Summary::Backlight);
        notifier.set_body(read::backlight(&mut icon));
        notifier.set_icon(&icon);
        notifier.send();
        return;
    }

    if opt.battery {
        notifier.set_summary(Summary::Battery);
        notifier.set_body(read::battery(&mut icon));
        notifier.set_icon(&icon);
        notifier.send();
        return;
    }

    if opt.network {
        let ifname = opt.interface.as_deref();
        notifier.set_summary(Summary::Network);
        notifier.set_body(read::network(ifname, &mut icon));
        notifier.set_icon(&icon);
        notifier.send();
        return;
    }

    if opt.memory {
        notifier.set_summary(Summary::Memory);
        notifier.set_body(read::memory(&mut icon));
        notifier.set_icon(&icon);
        notifier.send();
    }
}

pub fn get_started() {
    println!("{}", "// To get a network notification, run:".cyan());
    println!("citron -n -i <interface>");
    println!(
        "{}",
        "// To get a battery notification with a snazzy icon next to it, run:".cyan()
    );
    println!("citron -b -IT <icon-theme>");
    println!(
        "{}",
        "// To get a date notification in railway time, run:".cyan()
    );
    println!("citron -dR");
    println!("{}", "// For more information, run:".cyan());
    println!("citron -h");
}

pub fn get_version() {
    const VERSION: &str = env!("CARGO_PKG_VERSION");
    const PACKAGE: &str = env!("CARGO_PKG_NAME");
    println!("{} {}", PACKAGE.green(), VERSION);
}
