use super::Iconize;
use crate::Icon;
use libmacchina::traits::BatteryState;
use std::fmt;

pub struct BatteryFormatter {
    percentage: u8,
    state: BatteryState,
}

impl BatteryFormatter {
    pub fn new() -> Self {
        Self {
            percentage: 0,
            state: BatteryState::Discharging,
        }
    }

    pub fn set_percentage(&mut self, p: u8) {
        self.percentage = p;
    }

    pub fn set_state(&mut self, s: BatteryState) {
        self.state = s;
    }
}

impl Iconize for BatteryFormatter {
    fn set_icon(&self, i: &mut Icon) {
        match (self.percentage, &self.state) {
            (0..=10, BatteryState::Charging) => i.find_icon("battery-level-0-charging-symbolic"),
            (0..=10, BatteryState::Discharging) => i.find_icon("battery-level-0-symbolic"),
            (11..=20, BatteryState::Charging) => i.find_icon("battery-level-10-charging-symbolic"),
            (11..=20, BatteryState::Discharging) => i.find_icon("battery-level-10-symbolic"),
            (21..=30, BatteryState::Charging) => i.find_icon("battery-level-20-charging-symbolic"),
            (21..=30, BatteryState::Discharging) => i.find_icon("battery-level-20-symbolic"),
            (31..=40, BatteryState::Charging) => i.find_icon("battery-level-30-charging-symbolic"),
            (31..=40, BatteryState::Discharging) => i.find_icon("battery-level-30-symbolic"),
            (41..=50, BatteryState::Charging) => i.find_icon("battery-level-40-charging-symbolic"),
            (41..=50, BatteryState::Discharging) => i.find_icon("battery-level-40-symbolic"),
            (51..=60, BatteryState::Charging) => i.find_icon("battery-level-50-charging-symbolic"),
            (51..=60, BatteryState::Discharging) => i.find_icon("battery-level-50-symbolic"),
            (61..=70, BatteryState::Charging) => i.find_icon("battery-level-60-charging-symbolic"),
            (61..=70, BatteryState::Discharging) => i.find_icon("battery-level-60-symbolic"),
            (71..=80, BatteryState::Charging) => i.find_icon("battery-level-70-charging-symbolic"),
            (71..=80, BatteryState::Discharging) => i.find_icon("battery-level-70-symbolic"),
            (81..=90, BatteryState::Charging) => i.find_icon("battery-level-80-charging-symbolic"),
            (81..=90, BatteryState::Discharging) => i.find_icon("battery-level-80-symbolic"),
            (91..=99, BatteryState::Charging) => i.find_icon("battery-level-90-charging-symbolic"),
            (91..=99, BatteryState::Discharging) => i.find_icon("battery-level-90-symbolic"),
            _ => i.find_icon("battery-full-symbolic"),
        }
    }
}

impl fmt::Display for BatteryFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if self.percentage == 100 {
            return write!(f, "State: Full");
        }

        match self.state {
            BatteryState::Charging => {
                write!(f, "Percentage: {}%\nState: Charging", self.percentage)
            }
            BatteryState::Discharging => {
                write!(f, "Percentage: {}%\nState: Discharging", self.percentage)
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display_battery() {
        let mut fmt = BatteryFormatter::new();

        fmt.set_percentage(52);
        fmt.set_state(BatteryState::Charging);
        assert_eq!(
            fmt.to_string(),
            "Percentage: 52%\nState: Charging".to_owned()
        );

        fmt.set_percentage(100);
        assert_eq!(fmt.to_string(), "State: Full".to_owned());
    }
}
