//! This module is responsible for reading data from
//! [libmacchina](https://docs.rs/libmacchina/latest/libmacchina/) and sending it to its respective
//! formatter.

use libmacchina::traits::BatteryReadout as _;
use libmacchina::traits::GeneralReadout as _;
use libmacchina::traits::MemoryReadout as _;
use libmacchina::traits::NetworkReadout as _;
use libmacchina::traits::ReadoutError;
use libmacchina::{BatteryReadout, GeneralReadout, MemoryReadout, NetworkReadout};

use crate::fmt::backlight::BacklightFormatter;
use crate::fmt::battery::BatteryFormatter;
use crate::fmt::date::DateFormatter;
use crate::fmt::memory::MemoryFormatter;
use crate::fmt::network::NetworkFormatter;
use crate::fmt::uptime::UptimeFormatter;
use crate::helpers;
use crate::Icon;

use chrono::{Datelike, Timelike};

pub fn uptime() -> Result<String, ReadoutError> {
    let readout = GeneralReadout::new();
    let fmt = UptimeFormatter::from(readout.uptime()?);

    Ok(fmt.to_string())
}

pub fn network(interface: Option<&str>) -> Result<String, ReadoutError> {
    let mut fmt = NetworkFormatter::new();
    let readout = NetworkReadout::new();

    fmt.set_logical_address(readout.logical_address(interface)?);

    if let Ok(rx_bytes) = readout.rx_bytes(interface) {
        fmt.set_rx_bytes(rx_bytes);
    }

    if let Ok(tx_bytes) = readout.tx_bytes(interface) {
        fmt.set_tx_bytes(tx_bytes);
    }

    Ok(fmt.to_string())
}

pub fn memory() -> Result<String, ReadoutError> {
    let mut fmt = MemoryFormatter::new();
    let readout = MemoryReadout::new();

    fmt.set_total(readout.total()?);
    fmt.set_used(readout.used()?);
    fmt.set_percentage();

    Ok(fmt.to_string())
}

pub fn battery(icon: &mut Icon) -> Result<String, ReadoutError> {
    let mut fmt = BatteryFormatter::new();
    let readout = BatteryReadout::new();

    fmt.set_percentage(readout.percentage()?);
    fmt.set_state(readout.status()?);
    fmt.set_icon(icon);

    Ok(fmt.to_string())
}

pub fn backlight() -> Result<String, ReadoutError> {
    let mut fmt = BacklightFormatter::new();
    let readout = GeneralReadout::new();

    fmt.set_percentage(readout.backlight()?);

    Ok(fmt.to_string())
}

pub fn date(railway: bool, icon: &mut Icon) -> Result<String, ReadoutError> {
    let mut fmt = DateFormatter::new();
    if let Some(readout) = helpers::utc_from_timezone() {
        fmt.set_weekday(readout.weekday());
        fmt.set_month(readout.month());
        fmt.set_day(readout.day());
        fmt.set_minute(readout.minute());

        if railway {
            fmt.set_hour(readout.hour());
            fmt.enable_railway();
        } else {
            let h = readout.hour12();
            fmt.set_meridiem(h.0);
            fmt.set_hour(h.1);
        };

        fmt.set_icon(icon);

        return Ok(fmt.to_string());
    }

    Err(ReadoutError::Other(
        "I'm sorry! I couldn't retrieve your timezone.".to_string(),
    ))
}
