use crate::helpers;
use owo_colors::OwoColorize;
use std::path::{Path, PathBuf};

fn library_locations() -> Vec<PathBuf> {
    let mut locations = vec![];
    if cfg!(target_os = "linux") {
        locations.push("/lib");
    } else if cfg!(target_os = "netbsd") {
        locations.push("/usr/pkg/lib");
    } else if cfg!(target_family = "unix") {
        locations.push("/usr/local/lib");
    }

    locations.iter().map(|&x| PathBuf::from(x)).collect()
}

pub fn get_debug() {
    check_libdbus();
    check_daemon_capabilities();
    println!(
        "\nStill encountering issues?\nReport a 🐛 bug at {}",
        "https://github.com/grtcdr/citron/issues".blue()
    );
}

pub fn get_entries(path: &Path) -> Option<Vec<PathBuf>> {
    if let Ok(dir) = std::fs::read_dir(path) {
        let mut entries: Vec<PathBuf> = Vec::new();
        dir.flatten().for_each(|x| entries.push(x.path()));
        Some(entries)
    } else {
        None
    }
}

fn check_libdbus() {
    let base_locations = library_locations();

    let found = base_locations.iter().find(|&dir| {
        if let Some(libraries) = get_entries(dir) {
            return libraries.iter().any(|l| {
                let file = helpers::basename(l.to_str().unwrap_or_default());
                file.starts_with("libdbus-") && file.ends_with("so")
            });
        };

        false
    });

    if found.is_some() {
        println!("- Searching for libdbus: {}", "OK".green());
    } else {
        println!("- Searching for libdbus: {}", "FAILED".red());
    }
}

fn check_daemon_capabilities() {
    let capabilities = notify_rust::get_capabilities();
    if let Ok(capab) = capabilities {
        let mut list = vec![];
        println!("- Identifying your daemon's capabilities: {}", "OK".green());
        for c in capab {
            list.push(c);
        }
        println!("  * {}", list.join(", "));
    } else {
        println!(
            "- Identifying your daemon's capabilities: {}",
            "FAILED".red()
        );
        println!("  * Do you have a notification daemon running/installed?",);
        println!("  * Are you in a graphical session?");
    }
}
