use crate::Icon;
use libmacchina::traits::ReadoutError;
use notify_rust::Notification;
use notify_rust::NotificationHandle;
use notify_rust::Timeout;
use std::fmt;

pub struct Notifier {
    summary: Summary,
    timeout: u32,
    icon: String,
    body: String,
}

impl Notifier {
    pub fn new() -> Notifier {
        Notifier {
            summary: Summary::Date,
            timeout: 5,
            icon: String::new(),
            body: String::new(),
        }
    }

    pub fn set_summary(&mut self, k: Summary) {
        self.summary = k;
    }

    pub fn get_summary(&self) -> String {
        self.summary.to_string()
    }

    pub fn set_icon(&mut self, icon: &Icon) {
        if icon.is_visible() {
            if let Some(str) = icon.get_path().to_str() {
                self.icon = str.to_string();
            }
        }
    }

    pub fn set_timeout(&mut self, t: u32) {
        self.timeout = t;
    }

    pub fn get_timeout(&self) -> Timeout {
        Timeout::Milliseconds(self.timeout * 1000)
    }

    pub fn set_body(&mut self, r: Result<String, ReadoutError>) {
        match r {
            Ok(r) => self.body = r,
            Err(e) => self.body = e.to_string(),
        }
    }

    pub fn get_body(&self) -> String {
        self.body.to_owned()
    }

    pub fn send(&self) -> NotificationHandle {
        Notification::new()
            .body(&self.get_body())
            .summary(&self.get_summary())
            .timeout(self.get_timeout())
            .icon(&self.icon)
            .show()
            .expect("Verify that a notification daemon is running in the background")
    }
}

#[derive(Debug, PartialEq, Copy, Clone)]
pub enum Summary {
    Date,
    Uptime,
    Memory,
    Battery,
    Network,
    Backlight,
}

impl fmt::Display for Summary {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Summary::Date => write!(f, "Date"),
            Summary::Uptime => write!(f, "Uptime"),
            Summary::Memory => write!(f, "Memory"),
            Summary::Battery => write!(f, "Battery"),
            Summary::Network => write!(f, "Network"),
            Summary::Backlight => write!(f, "Backlight"),
        }
    }
}
