use crate::notifier::Summary;
use linicon::lookup_icon;
use owo_colors::OwoColorize;
use std::path::PathBuf;

pub struct Icon {
    path: PathBuf,
    theme: String,
    location: PathBuf,
    visible: bool,
}

impl Icon {
    pub fn new() -> Self {
        let base = if cfg!(target_os = "linux") {
            PathBuf::from("/usr/share/icons")
        } else if cfg!(target_os = "netbsd") {
            PathBuf::from("/usr/pkg/share/icons")
        } else {
            PathBuf::new()
        };

        Icon {
            path: PathBuf::new(),
            theme: String::new(),
            location: base,
            visible: false,
        }
    }

    pub fn get_location(&self) -> &str {
        self.location.to_str().unwrap_or_default()
    }

    pub fn is_visible(&self) -> bool {
        self.visible
    }

    pub fn get_path(&self) -> PathBuf {
        self.path.to_owned()
    }

    pub fn get_available_themes(&self) -> Vec<String> {
        let mut themes = vec![];
        let mut places = vec![self.location.clone()];

        if let Ok(home) = std::env::var("HOME") {
            places.push(PathBuf::from(home).join(".icons"));
        }

        for p in places {
            if let Ok(entries) = p.read_dir() {
                for entry in entries.flatten() {
                    if !entry.path().join("index.theme").exists() {
                        continue;
                    }

                    if let Some(name) = entry.path().file_name() {
                        themes.push(name.to_string_lossy().to_string());
                    }
                }
            }
        }

        themes
    }

    pub fn set_theme(&mut self, theme: String) {
        self.theme = theme;
    }

    pub fn make_visible(&mut self) {
        self.visible = true;
    }

    pub fn find_icon(&mut self, icon_name: &str) {
        if !self.visible {
            return;
        }

        if let Ok(icon) = lookup_icon(icon_name).with_search_paths(&[self.get_location()]) {
            if let Some(icon) = icon.from_theme(&self.theme).next() {
                match icon {
                    Ok(i) => self.path = i.path,
                    _ => {
                        // The requested icon theme does not exist.
                        eprintln!(
                            "{}: {} doesn't exist.",
                            "Warning".yellow().bold(),
                            self.theme.cyan()
                        );
                        // Print a list of usable themes.
                        eprintln!("Pick a theme from this list:");
                        self.get_available_themes()
                            .into_iter()
                            .filter(|x| !x.eq("default"))
                            .for_each(|x| eprintln!(" - {}", x.yellow()));
                    }
                }
            } else {
                // The requested icon does not exist for the given theme.
                eprintln!(
                    "{}: {} does not exist for {}.",
                    "Warning".yellow().bold(),
                    icon_name.purple(),
                    self.theme.cyan()
                )
            }
        }
    }

    pub fn set_icon_contextually(&mut self, summary: Summary) {
        if !self.visible {
            return;
        }

        match summary {
            Summary::Battery => {
                self.find_icon("battery-symbolic");
            }
            Summary::Date => {
                self.find_icon("weather-clear-symbolic");
            }
            Summary::Uptime => {
                self.find_icon("computer-symbolic");
            }
            Summary::Network => {
                self.find_icon("network-wireless-symbolic");
            }
            Summary::Backlight => {
                self.find_icon("display-brightness-symbolic");
            }
            Summary::Memory => {
                self.find_icon("media-floppy-symbolic");
            }
        }
    }

    pub fn set_icon(&mut self, name: &str) {
        self.find_icon(name)
    }
}
