//! This module is responsible for reading data from
//! [libmacchina](https://docs.rs/libmacchina/latest/libmacchina/) and sending it to its respective
//! formatter.

use libmacchina::traits::BatteryReadout as _;
use libmacchina::traits::GeneralReadout as _;
use libmacchina::traits::MemoryReadout as _;
use libmacchina::traits::NetworkReadout as _;
use libmacchina::traits::ReadoutError;
use libmacchina::{BatteryReadout, GeneralReadout, MemoryReadout, NetworkReadout};

use crate::fmt::backlight::BacklightFormatter;
use crate::fmt::battery::BatteryFormatter;
use crate::fmt::date::DateFormatter;
use crate::fmt::memory::MemoryFormatter;
use crate::fmt::network::NetworkFormatter;
use crate::fmt::uptime::UptimeFormatter;
use crate::helpers;

use chrono::{Datelike, Timelike, Utc};

pub fn uptime() -> Result<String, ReadoutError> {
    let readout = GeneralReadout::new();
    let fmt = UptimeFormatter::from(readout.uptime()?);

    Ok(fmt.to_string())
}

pub fn network(interface: Option<&str>) -> Result<String, ReadoutError> {
    let mut fmt = NetworkFormatter::new();
    let readout = NetworkReadout::new();

    fmt.set_logical_address(readout.logical_address(interface)?);
    fmt.set_rx_bytes(readout.rx_bytes(interface)?);
    fmt.set_tx_bytes(readout.tx_bytes(interface)?);

    Ok(fmt.to_string())
}

pub fn memory() -> Result<String, ReadoutError> {
    let mut fmt = MemoryFormatter::new();
    let readout = MemoryReadout::new();

    fmt.set_total(readout.total()?);
    fmt.set_used(readout.used()?);
    fmt.set_percentage();

    Ok(fmt.to_string())
}

pub fn battery() -> Result<String, ReadoutError> {
    let mut fmt = BatteryFormatter::new();
    let readout = BatteryReadout::new();

    fmt.set_percentage(readout.percentage()?);
    fmt.set_state(readout.status()?);

    Ok(fmt.to_string())
}

pub fn backlight() -> Result<String, ReadoutError> {
    let mut fmt = BacklightFormatter::new();
    let readout = GeneralReadout::new();

    fmt.set_percentage(readout.backlight()?);

    Ok(fmt.to_string())
}

pub fn date() -> Result<String, ReadoutError> {
    let mut fmt = DateFormatter::new();

    if let Some(timezone) = helpers::get_timezone() {
        let timezone = chrono_tz::TZ_VARIANTS
            .iter()
            .find(|&v| *v.to_string() == timezone);

        if let Some(tz) = timezone {
            let readout = Utc::now().with_timezone(tz);
            fmt.set_day(readout.day());
            fmt.set_hour(readout.hour12());
            fmt.set_month(readout.month());
            fmt.set_weekday(readout.weekday());
            fmt.set_minute(readout.minute());

            return Ok(fmt.to_string());
        }
    }

    Err(ReadoutError::Other(
        "I'm sorry! I couldn't retrieve your timezone.".to_string(),
    ))
}
