#![allow(unused_must_use)]

// The flow of citron is as follows:
//
//   read   ---->   format   ---->   send
//    |              |                |
//    |              |                |
//    |              |                `---> `send` module
//    |              |
//    |              `---> `fmt` modules
//    |
//    `--> `read` module

mod cli;
mod debug;
mod fmt;
mod goodies;
mod helpers;
mod notifier;
mod read;
mod send;

use notifier::{Notifier, Summary};
use send::send;
use structopt::StructOpt;

fn main() {
    let opt = cli::Opt::from_args();
    let mut notifier = Notifier::new();

    if let Some(timeout) = opt.timeout {
        notifier.set_timeout(timeout)
    }

    if opt.version {
        get_version();
        return;
    }

    if opt.debug {
        debug::get_debug();
        return;
    }

    if opt.uptime {
        notifier.set_summary(Summary::Uptime);
        send(read::uptime(), notifier);
        return;
    }

    if opt.date {
        notifier.set_summary(Summary::Date);
        send(read::date(), notifier);
        return;
    }

    if opt.backlight {
        notifier.set_summary(Summary::Backlight);
        send(read::backlight(), notifier);
        return;
    }

    if opt.battery {
        notifier.set_summary(Summary::Battery);
        send(read::battery(), notifier);
        return;
    }

    if opt.network {
        notifier.set_summary(Summary::Network);
        send(read::network(opt.interface.as_deref()), notifier);
        return;
    }

    if opt.memory {
        notifier.set_summary(Summary::Memory);
        send(read::memory(), notifier);
    }
}

pub fn get_version() {
    const VERSION: &str = env!("CARGO_PKG_VERSION");
    const PACKAGE: &str = env!("CARGO_PKG_NAME");
    println!("{} {}", PACKAGE, VERSION);
}
