use bytesize::ByteSize;
use std::fmt;

pub struct NetworkFormatter {
    logical_address: String,
    rx_bytes: usize,
    tx_bytes: usize,
}

impl NetworkFormatter {
    pub fn new() -> Self {
        Self {
            logical_address: String::from("N/A"),
            rx_bytes: 0,
            tx_bytes: 0,
        }
    }

    pub fn set_logical_address(&mut self, addr: String) {
        self.logical_address = addr;
    }

    pub fn set_rx_bytes(&mut self, bytes: usize) {
        self.rx_bytes = bytes;
    }

    pub fn set_tx_bytes(&mut self, bytes: usize) {
        self.tx_bytes = bytes;
    }

    pub fn get_rx_bytes(&self) -> ByteSize {
        ByteSize::kb(self.rx_bytes as u64 / 1024)
    }

    pub fn get_tx_bytes(&self) -> ByteSize {
        ByteSize::kb(self.tx_bytes as u64 / 1024)
    }
}

impl fmt::Display for NetworkFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if self.rx_bytes == 0 && self.tx_bytes == 0 {
            return write!(f, "Local IP: {}", self.logical_address);
        }

        write!(
            f,
            "Local IP: {}\nData: {} 🠗 / {} 🠕",
            self.logical_address,
            self.get_rx_bytes(),
            self.get_tx_bytes(),
        )
    }
}
