use crate::goodies::{Battery, Month, Time};
use bytesize::ByteSize;
use chrono::{Datelike, Timelike, Utc};
use libmacchina::traits::BatteryReadout as _;
use libmacchina::traits::GeneralReadout as _;
use libmacchina::traits::MemoryReadout as _;
use libmacchina::traits::ReadoutError;
use libmacchina::BatteryReadout;
use libmacchina::GeneralReadout;
use libmacchina::MemoryReadout;

pub fn uptime() -> Result<Time, ReadoutError> {
    match GeneralReadout::new().uptime() {
        Ok(up) => Ok(Time::from(up)),
        _ => Err(ReadoutError::Warning(
            "I couldn't get this statistic :(".to_string(),
        )),
    }
}

pub fn network(interface: Option<String>) -> Result<String, ReadoutError> {
    match GeneralReadout::new().local_ip(interface) {
        Ok(ip) => Ok(format!("Local IP: {}", ip)),
        Err(e) => Err(ReadoutError::Warning(e.to_string())),
    }
}

pub fn memory() -> Result<String, ReadoutError> {
    let read = MemoryReadout::new();
    match (read.used(), read.total()) {
        (Ok(u), Ok(t)) => {
            let total = ByteSize::kb(t);
            let used = ByteSize::kb(u);
            let percentage = u as f64 / t as f64 * 100f64;

            Ok(format!("{}/{} ({:.1}%)", used, total, percentage))
        }
        _ => Err(ReadoutError::Warning(
            "I couldn't get this statistic :(".to_string(),
        )),
    }
}

pub fn battery() -> Result<String, ReadoutError> {
    let read = BatteryReadout::new();
    match (read.percentage(), read.status()) {
        (Ok(p), Ok(s)) => Ok(Battery::new(p, s).to_string()),
        _ => Err(ReadoutError::Warning(
            "I couldn't get this statistic :(".to_string(),
        )),
    }
}

pub fn date_time() -> Result<String, ReadoutError> {
    let now = Utc::now();
    let (is_pm, hour) = now.hour12();
    let date = format!(
        "{}, {} {:02}",
        now.weekday(),
        Month::from(now.month()),
        now.day()
    );
    let time = format!(
        "{:02}:{:02} {}",
        hour,
        now.minute(),
        if is_pm { "PM" } else { "AM" }
    );

    Ok(format!("{} at {}", date, time))
}
