use crate::options::Subcommand;
use std::path::PathBuf;

use libmacchina::traits::ReadoutError;
use notify_rust::Notification;
use notify_rust::Timeout;

use std::fmt;

pub struct Notifier {
    /// The header of a notification.
    pub title: Title,
    /// The body of a notification.
    pub body: String,
    /// The path to an icon.
    pub icon: PathBuf,
    /// The timeout of a notification.
    pub timeout: u32,
}

#[allow(clippy::new_without_default)]
impl Notifier {
    pub fn new() -> Notifier {
        Notifier {
            title: Title::Date,
            timeout: 5,
            icon: PathBuf::new(),
            body: String::new(),
        }
    }

    pub fn set_body(&mut self, body: Result<String, ReadoutError>) {
        match body {
            Ok(b) => self.body = b,
            Err(e) => self.body = e.to_string(),
        }
    }

    pub fn send(&self) {
        Notification::new()
            .body(&self.body)
            .summary(&self.title.to_string())
            .timeout(Timeout::Milliseconds(self.timeout * 1000))
            .icon(&self.icon.to_string_lossy())
            .show()
            .expect("No notification daemon running.");
    }
}

#[derive(Debug, PartialEq, Clone)]
pub enum Title {
    Date,
    Uptime,
    Memory,
    Battery,
    Network,
    Backlight,
    Custom(String),
}

impl fmt::Display for Title {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match &*self {
            Title::Date => write!(f, "Date"),
            Title::Uptime => write!(f, "Uptime"),
            Title::Memory => write!(f, "Memory"),
            Title::Battery => write!(f, "Battery"),
            Title::Network => write!(f, "Network"),
            Title::Backlight => write!(f, "Backlight"),
            Title::Custom(s) => write!(f, "{}", s),
        }
    }
}

impl From<&Subcommand> for Title {
    fn from(cmd: &Subcommand) -> Self {
        match cmd {
            Subcommand::Date => Title::Date,
            Subcommand::Uptime => Title::Uptime,
            Subcommand::Memory => Title::Memory,
            Subcommand::Battery => Title::Battery,
            Subcommand::Network => Title::Network,
            Subcommand::Backlight => Title::Backlight,
        }
    }
}
