#![allow(unused_must_use)]

mod config;
mod fmt;

mod icon;
mod notifier;
mod options;
mod read;

use config::Config;
use icon::Icon;
use notifier::{Notifier, Title};
use options::{Options, Subcommand};
use structopt::StructOpt;

pub enum Stream {
    Stderr,
    Stdout,
}

fn main() {
    let mut notifier = Notifier::new();
    let mut icon = Icon::new();
    let opt = Options::from_args();
    let config = Config::read_config().unwrap_or_default();

    if opt.icon_themes {
        if let Err(e) = icon.print_available_themes(Stream::Stdout) {
            eprintln!("Error: {}", e);
        }
    }

    if let Some(ico) = &config.icon {
        if let Some(theme) = &ico.theme {
            icon.theme = Some(theme.to_string());
        }
    }

    let use_icons = icon.theme.is_some();

    if let Some(cmd) = &opt.cmd {
        notifier.title = Title::from(cmd);
        match cmd {
            Subcommand::Backlight => {
                notifier.set_body(read::backlight((&mut icon.name, use_icons)));
                if let Some(bac) = &config.backlight {
                    if let Some(ti) = &bac.title {
                        notifier.title = Title::Custom(ti.to_string());
                    }
                }
            }
            Subcommand::Battery => {
                notifier.set_body(read::battery(&config, (&mut icon.name, use_icons)));
                if let Some(bat) = &config.battery {
                    if let Some(ti) = &bat.title {
                        notifier.title = Title::Custom(ti.to_string());
                    }
                }
            }
            Subcommand::Uptime => {
                notifier.set_body(read::uptime((&mut icon.name, use_icons)));
                if let Some(up) = &config.uptime {
                    if let Some(ti) = &up.title {
                        notifier.title = Title::Custom(ti.to_string());
                    }
                }
            }
            Subcommand::Network => {
                notifier.set_body(read::network(&config, (&mut icon.name, use_icons)));
                if let Some(net) = &config.network {
                    if let Some(ti) = &net.title {
                        notifier.title = Title::Custom(ti.to_string());
                    }
                }
            }
            Subcommand::Date => {
                notifier.set_body(read::date(&config, (&mut icon.name, use_icons)));
                if let Some(date) = &config.date {
                    if let Some(ti) = &date.title {
                        notifier.title = Title::Custom(ti.to_string());
                    }
                }
            }
            Subcommand::Memory => {
                notifier.set_body(read::memory((&mut icon.name, use_icons)));
                if let Some(mem) = &config.memory {
                    if let Some(ti) = &mem.title {
                        notifier.title = Title::Custom(ti.to_string());
                    }
                }
            }
        }

        match (icon.find_icon(), use_icons) {
            (Ok(path), true) => notifier.icon = path,
            (Err(e), true) => eprintln!("{}", e),
            _ => (),
        }

        if let Some(timeout) = config.timeout {
            notifier.timeout = timeout;
        }

        notifier.send();
    }
}
