use super::{Format, Iconize};
use bytesize::ByteSize;
use std::fmt;

pub struct NetworkFormatter {
    logical_address: Option<String>,
    rx_bytes: Option<usize>,
    tx_bytes: Option<usize>,
}

impl NetworkFormatter {
    pub fn new() -> Self {
        Self {
            logical_address: None,
            rx_bytes: None,
            tx_bytes: None,
        }
    }

    pub fn set_logical_address(&mut self, addr: String) {
        self.logical_address = Some(addr)
    }

    pub fn get_logical_address(&self) -> Option<String> {
        self.logical_address.to_owned()
    }

    pub fn set_rx_bytes(&mut self, bytes: usize) {
        self.rx_bytes = Some(bytes);
    }

    pub fn set_tx_bytes(&mut self, bytes: usize) {
        self.tx_bytes = Some(bytes);
    }

    pub fn get_rx_bytes(&self) -> Option<ByteSize> {
        self.rx_bytes.map(|bytes| ByteSize::kb(bytes as u64 / 1024))
    }

    pub fn get_tx_bytes(&self) -> Option<ByteSize> {
        self.tx_bytes.map(|bytes| ByteSize::kb(bytes as u64 / 1024))
    }
}

impl Iconize for NetworkFormatter {
    fn set_icon(&self) -> &str {
        match self.logical_address {
            Some(_) => "network-wireless-symbolic",
            _ => "network-wireless-disconnected-symbolic",
        }
    }
}

impl Format for NetworkFormatter {
    fn with_format(&self, mut format: String) -> String {
        match self.get_logical_address() {
            Some(addr) => format = format.replace("%a", &addr),
            _ => format = format.replace("%a", ""),
        }

        match self.get_rx_bytes() {
            Some(bytes) => format = format.replace("%r", &bytes.to_string()),
            _ => format = format.replace("%r", ""),
        }

        match self.get_tx_bytes() {
            Some(bytes) => format = format.replace("%t", &bytes.to_string()),
            _ => format = format.replace("%t", ""),
        }

        format
    }
}

impl fmt::Display for NetworkFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if let Some(addr) = &self.get_logical_address() {
            writeln!(f, "Local IP: {}", addr);
        }

        match (self.get_rx_bytes(), self.get_tx_bytes()) {
            (Some(rx), Some(tx)) => write!(f, "Data: {} 🠗 / {} 🠕", rx, tx),
            _ => Ok(()),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display() {
        let mut fmt = NetworkFormatter::new();

        fmt.set_logical_address("127.0.0.1".to_owned());
        assert_eq!(fmt.to_string(), "Local IP: 127.0.0.1\n".to_owned());

        fmt.set_logical_address("192.168.1.1".to_owned());
        fmt.set_rx_bytes(717000);
        fmt.set_tx_bytes(717000);
        assert_eq!(
            fmt.to_string(),
            "Local IP: 192.168.1.1\nData: 700.0 KB 🠗 / 700.0 KB 🠕".to_owned()
        );
    }

    #[test]
    fn display_with_format() {
        let mut fmt = NetworkFormatter::new();
        fmt.set_rx_bytes(717000);
        let f = String::from("%r");
        assert_eq!(fmt.with_format(f), "700.0 KB");

        fmt.set_logical_address("192.168.1.1".to_owned());
        fmt.set_tx_bytes(717000);
        let f = String::from("%a, %r, %t");
        assert_eq!(fmt.with_format(f), "192.168.1.1, 700.0 KB, 700.0 KB");
    }

    #[test]
    fn pick_icon() {
        let mut fmt = NetworkFormatter::new();
        assert_eq!(fmt.set_icon(), "network-wireless-disconnected-symbolic");
        fmt.set_logical_address("192.168.1.1".to_string());
        assert_eq!(fmt.set_icon(), "network-wireless-symbolic");
    }
}
