//! This module is responsible for reading data from
//! [libmacchina](https://docs.rs/libmacchina/latest/libmacchina/) and sending it to its respective
//! formatter.

use crate::config::Config;
use crate::fmt::backlight::BacklightFormatter;
use crate::fmt::battery::BatteryFormatter;
use crate::fmt::date::DateFormatter;
use crate::fmt::memory::MemoryFormatter;
use crate::fmt::network::NetworkFormatter;
use crate::fmt::uptime::UptimeFormatter;
use crate::fmt::{Format, Iconize};
use crate::helpers;
use chrono::{Datelike, Timelike};
use libmacchina::traits::BatteryReadout as _;
use libmacchina::traits::GeneralReadout as _;
use libmacchina::traits::MemoryReadout as _;
use libmacchina::traits::NetworkReadout as _;
use libmacchina::traits::ReadoutError;
use libmacchina::{BatteryReadout, GeneralReadout, MemoryReadout, NetworkReadout};

pub fn uptime(icon: (&mut String, bool)) -> Result<String, ReadoutError> {
    let readout = GeneralReadout::new();
    let fmt = UptimeFormatter::from(readout.uptime()?);

    if icon.1 {
        icon.0.push_str(fmt.set_icon());
    }

    Ok(fmt.to_string())
}

pub fn network(config: &Config, icon: (&mut String, bool)) -> Result<String, ReadoutError> {
    let mut fmt = NetworkFormatter::new();
    let readout = NetworkReadout::new();

    if let Some(network) = &config.network {
        if let Some(interface) = &network.interface {
            let ifname = Some(interface.as_str());

            match readout.logical_address(ifname) {
                Ok(addr) => {
                    if icon.1 {
                        icon.0.push_str(fmt.set_icon());
                    }
                    fmt.set_logical_address(addr);
                }
                Err(e) => {
                    if icon.1 {
                        icon.0.push_str(fmt.set_icon());
                    }
                    return Err(ReadoutError::Other(e.to_string()));
                }
            }

            if let Ok(bytes) = readout.rx_bytes(ifname) {
                fmt.set_rx_bytes(bytes);
            }

            if let Ok(tx_bytes) = readout.tx_bytes(ifname) {
                fmt.set_tx_bytes(tx_bytes);
            }

            if let Some(format) = &network.format {
                return Ok(fmt.with_format(format.to_string()));
            }
        }
    }

    Ok(fmt.to_string())
}

pub fn memory(icon: (&mut String, bool)) -> Result<String, ReadoutError> {
    let mut fmt = MemoryFormatter::new();
    let readout = MemoryReadout::new();

    fmt.set_total(readout.total()?);
    fmt.set_used(readout.used()?);
    fmt.set_percentage();

    if icon.1 {
        icon.0.push_str(fmt.set_icon());
    }

    Ok(fmt.to_string())
}

pub fn battery(config: &Config, icon: (&mut String, bool)) -> Result<String, ReadoutError> {
    let mut fmt = BatteryFormatter::new();
    let readout = BatteryReadout::new();

    match readout.percentage() {
        Ok(p) => fmt.set_percentage(p),
        Err(e) => eprintln!("{}", e.to_string()),
    }

    match readout.status() {
        Ok(s) => fmt.set_state(s),
        Err(e) => eprintln!("{}", e.to_string()),
    }

    if icon.1 {
        icon.0.push_str(fmt.set_icon());
    }

    if let Some(battery) = &config.battery {
        if let Some(format) = &battery.format {
            return Ok(fmt.with_format(format.to_owned()));
        }
    }

    Ok(fmt.to_string())
}

pub fn backlight(icon: (&mut String, bool)) -> Result<String, ReadoutError> {
    let mut fmt = BacklightFormatter::new();
    let readout = GeneralReadout::new();

    fmt.set_percentage(readout.backlight()?);

    if icon.1 {
        icon.0.push_str(fmt.set_icon());
    }

    Ok(fmt.to_string())
}

pub fn date(config: &Config, icon: (&mut String, bool)) -> Result<String, ReadoutError> {
    let mut fmt = DateFormatter::new();
    if let Some(readout) = helpers::utc_from_timezone() {
        let hour = readout.hour12();
        fmt.set_weekday(readout.weekday());
        fmt.set_month(readout.month());
        fmt.set_day(readout.day());
        fmt.set_minute(readout.minute());
        fmt.set_meridiem(hour.0);
        fmt.set_hour(hour.1);

        if icon.1 {
            icon.0.push_str(fmt.set_icon());
        }

        if let Some(date) = &config.date {
            if date.is_railway() {
                fmt.set_hour(readout.hour());
            }

            if let Some(format) = &date.format {
                return Ok(fmt.with_format(format.to_owned()));
            }
        }

        return Ok(fmt.to_string());
    }

    Err(ReadoutError::Other(
        "Oops, we couldn't retrieve your timezone.".to_string(),
    ))
}
