use crate::fmt::date::{Day, Meridiem, Month};
use crate::fmt::{Format, Iconize};
use chrono::Weekday;
use std::fmt;

#[derive(Debug, PartialEq)]
pub struct DateFormatter {
    hour: u32,
    minute: u32,
    day: u32,
    month: Month,
    railway: bool,
    weekday: Day,
    meridiem: Meridiem,
}

impl DateFormatter {
    pub fn new() -> Self {
        Self {
            hour: 0,
            minute: 0,
            day: 0,
            month: Month::January,
            railway: false,
            meridiem: Meridiem::Ante,
            weekday: Day::Monday,
        }
    }

    pub fn set_hour(&mut self, hour: u32) {
        self.hour = hour;
    }

    #[cfg(test)]
    pub fn use_railway(&mut self) {
        self.railway = true;
    }

    pub fn set_minute(&mut self, minute: u32) {
        self.minute = minute;
    }

    pub fn set_day(&mut self, day: u32) {
        self.day = day;
    }

    pub fn set_month(&mut self, month: u32) {
        self.month = month.into();
    }

    pub fn set_weekday(&mut self, weekday: Weekday) {
        self.weekday = weekday.into();
    }

    pub fn set_meridiem(&mut self, pm: bool) {
        if pm {
            self.meridiem = Meridiem::Post;
        }
    }
}

impl Iconize for DateFormatter {
    fn set_icon(&self) -> &str {
        let night = if self.railway {
            self.hour >= 20 || self.hour <= 6
        } else {
            self.hour >= 8 && self.meridiem == Meridiem::Post
                || self.hour <= 6 && self.meridiem == Meridiem::Ante
        };

        if night {
            "weather-clear-night-symbolic"
        } else {
            "weather-clear-symbolic"
        }
    }
}

impl Format for DateFormatter {
    fn with_format(&self, format: String) -> String {
        format
            .replace("%HH", &format!("{:0>2}", &self.hour))
            .replace("%MM", &format!("{:0>2}", &self.minute))
            .replace("%P", &self.meridiem.to_string())
            .replace("%w", &self.weekday.short().to_string())
            .replace("%W", &self.weekday.long().to_string())
            .replace("%m", &self.month.short().to_string())
            .replace("%M", &self.month.long().to_string())
            .replace("%d", &self.day.to_string())
    }
}

impl fmt::Display for DateFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if self.railway {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02}",
                self.weekday.short(),
                self.month.short(),
                self.day,
                self.hour,
                self.minute,
            )
        } else {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02} {}",
                self.weekday.short(),
                self.month.short(),
                self.day,
                self.hour,
                self.minute,
                self.meridiem,
            )
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display() {
        let mut fmt = DateFormatter::new();
        fmt.set_weekday(Weekday::Sat);
        fmt.set_meridiem(true);
        fmt.set_day(1);
        fmt.set_month(8);
        fmt.set_hour(2);
        fmt.set_minute(37);
        assert_eq!(fmt.to_string(), "Sat, Aug 01 at 02:37 PM".to_owned());

        fmt.use_railway();
        fmt.set_weekday(Weekday::Mon);
        fmt.set_day(13);
        fmt.set_month(9);
        fmt.set_hour(16);
        fmt.set_minute(12);
        assert_eq!(fmt.to_string(), "Mon, Sep 13 at 16:12".to_owned());
    }

    #[test]
    fn display_with_format() {
        let mut fmt = DateFormatter::new();

        fmt.set_weekday(Weekday::Mon);
        let f = String::from("%w");
        assert_eq!(fmt.with_format(f), "Mon".to_string());

        fmt.set_weekday(Weekday::Sat);
        let f = String::from("%W");
        assert_eq!(fmt.with_format(f), "Saturday".to_string());

        fmt.set_month(6);
        let f = String::from("%m");
        assert_eq!(fmt.with_format(f), "Jun".to_string());

        fmt.set_month(7);
        let f = String::from("%M");
        assert_eq!(fmt.with_format(f), "July".to_string());

        fmt.set_day(12);
        let f = String::from("%d");
        assert_eq!(fmt.with_format(f), "12".to_string());

        fmt.set_hour(20);
        fmt.set_minute(20);
        let f = String::from("%HH:%MM");
        assert_eq!(fmt.with_format(f), "20:20".to_string());

        fmt.set_hour(12);
        fmt.set_minute(20);
        fmt.set_meridiem(false);
        let f = String::from("%HH:%MM %P");
        assert_eq!(fmt.with_format(f), "12:20 AM".to_string());
    }

    #[test]
    fn pick_icon() {
        let mut fmt = DateFormatter::new();
        fmt.set_weekday(Weekday::Sun);
        fmt.set_meridiem(true);
        fmt.set_day(1);
        fmt.set_month(8);
        fmt.set_hour(2);
        fmt.set_minute(37);
        assert_eq!(fmt.set_icon(), "weather-clear-symbolic");
    }
}
