use chrono::{DateTime, Utc};
use chrono_tz::Tz;
use std::fs;

pub fn get_timezone() -> Option<String> {
    if let Ok(link) = fs::read_link("/etc/localtime") {
        let mut components = link.components();
        let country = components.next_back();
        let region = components.next_back();
        return match (region, country) {
            (Some(reg), Some(cou)) => Some(format!(
                "{}/{}",
                reg.as_os_str().to_string_lossy(),
                cou.as_os_str().to_string_lossy()
            )),
            _ => None,
        };
    } else {
        let timezone = std::fs::read_to_string("/etc/timezone");
        if let Ok(str) = timezone {
            return Some(str);
        }
    }

    None
}

pub fn utc_from_timezone() -> Option<DateTime<Tz>> {
    if let Some(tz) = get_timezone() {
        let timezone = chrono_tz::TZ_VARIANTS.iter().find(|&x| x.to_string() == tz);

        timezone.map(|tz| Utc::now().with_timezone(tz))
    } else {
        None
    }
}
