use super::{Format, Iconize};
use libmacchina::traits::BatteryState;
use std::fmt;

pub struct BatteryFormatter {
    percentage: Option<u8>,
    state: Option<BatteryState>,
}

impl BatteryFormatter {
    pub fn new() -> Self {
        Self {
            percentage: None,
            state: None,
        }
    }

    pub fn set_percentage(&mut self, p: u8) {
        self.percentage = Some(p);
    }

    pub fn get_percentage(&self) -> Option<String> {
        self.percentage.map(|p| format!("{}%", p))
    }

    pub fn get_state(&self) -> Option<String> {
        self.state.as_ref().map(|s| s.to_string())
    }

    pub fn set_state(&mut self, s: BatteryState) {
        self.state = Some(s);
    }
}

impl Iconize for BatteryFormatter {
    fn set_icon(&self) -> &str {
        match (self.percentage, &self.state) {
            (Some(0..=10), Some(BatteryState::Charging)) => "battery-level-0-charging-symbolic",
            (Some(0..=10), Some(BatteryState::Discharging)) => "battery-level-0-symbolic",
            (Some(11..=20), Some(BatteryState::Charging)) => "battery-level-10-charging-symbolic",
            (Some(11..=20), Some(BatteryState::Discharging)) => "battery-level-10-symbolic",
            (Some(21..=30), Some(BatteryState::Charging)) => "battery-level-20-charging-symbolic",
            (Some(21..=30), Some(BatteryState::Discharging)) => "battery-level-20-symbolic",
            (Some(31..=40), Some(BatteryState::Charging)) => "battery-level-30-charging-symbolic",
            (Some(31..=40), Some(BatteryState::Discharging)) => "battery-level-30-symbolic",
            (Some(41..=50), Some(BatteryState::Charging)) => "battery-level-40-charging-symbolic",
            (Some(41..=50), Some(BatteryState::Discharging)) => "battery-level-40-symbolic",
            (Some(51..=60), Some(BatteryState::Charging)) => "battery-level-50-charging-symbolic",
            (Some(51..=60), Some(BatteryState::Discharging)) => "battery-level-50-symbolic",
            (Some(61..=70), Some(BatteryState::Charging)) => "battery-level-60-charging-symbolic",
            (Some(61..=70), Some(BatteryState::Discharging)) => "battery-level-60-symbolic",
            (Some(71..=80), Some(BatteryState::Charging)) => "battery-level-70-charging-symbolic",
            (Some(71..=80), Some(BatteryState::Discharging)) => "battery-level-70-symbolic",
            (Some(81..=90), Some(BatteryState::Charging)) => "battery-level-80-charging-symbolic",
            (Some(81..=90), Some(BatteryState::Discharging)) => "battery-level-80-symbolic",
            (Some(91..=99), Some(BatteryState::Charging)) => "battery-level-90-charging-symbolic",
            (Some(91..=99), Some(BatteryState::Discharging)) => "battery-level-90-symbolic",
            (Some(_), Some(_)) => "battery-full-symbolic",
            _ => "battery-missing-symbolic",
        }
    }
}

impl Format for BatteryFormatter {
    fn with_format(&self, format: String) -> String {
        if self.percentage.is_none() && self.percentage.is_none() {
            return self.to_string();
        }

        format
            .replace("%p", &self.get_percentage().unwrap_or_else(String::new))
            .replace("%s", &self.get_state().unwrap_or_else(String::new))
    }
}

impl fmt::Display for BatteryFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if let Some(p) = self.percentage {
            if p == 100 {
                return write!(f, "State: Full");
            }

            writeln!(f, "Percentage: {}%", p);

            if let Some(s) = &self.state {
                return write!(f, "State: {}", s);
            }

            return write!(f, "State: Unknown");
        }

        return write!(f, "No battery detected.");
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display() {
        let mut fmt = BatteryFormatter::new();
        assert_eq!(fmt.to_string(), "No battery detected.".to_owned());

        fmt.set_percentage(50);
        assert_eq!(fmt.to_string(), "Percentage: 50%\nState: Unknown".to_owned());

        fmt.set_state(BatteryState::Charging);
        assert_eq!(
            fmt.to_string(),
            "Percentage: 50%\nState: Charging".to_owned()
        );

        fmt.set_percentage(100);
        assert_eq!(fmt.to_string(), "State: Full".to_owned());
    }

    #[test]
    fn display_with_format() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(80);
        fmt.set_state(BatteryState::Discharging);
        let f = String::from("%p\n%s");
        assert_eq!(fmt.with_format(f), "80%\nDischarging".to_owned());
    }

    #[test]
    fn pick_icon() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(85);
        fmt.set_state(BatteryState::Discharging);
        assert_eq!(fmt.set_icon(), "battery-level-80-symbolic");
    }
}
