#![allow(unused_must_use)]

// The flow of citron is as follows:
//
//   read   ---->   format   ---->   notify
//    |              |                |
//    |              |                |
//    |              |                `---> `notifier` module
//    |              |
//    |              `---> `fmt` modules
//    |
//    `--> `read` module

mod config;
mod debug;
mod fmt;
mod helpers;
mod icon;
mod notifier;
mod options;
mod read;

use config::Config;
use debug::get_debug;
use icon::Icon;
use notifier::{Notifier, Title};
use options::Command;
use options::Options;
use structopt::StructOpt;

fn main() {
    let mut notifier = Notifier::new();
    let mut icon = Icon::new();
    let opt = Options::from_args();
    let config = match Config::read_config() {
        Some(c) => c,
        None => Config::default(),
    };

    if opt.debug {
        get_debug();
        return;
    }

    if opt.icon_themes {
        println!("Available icon themes:");
        icon.get_available_themes()
            .into_iter()
            .filter(|x| !x.eq("default"))
            .for_each(|x| eprintln!(" > {}", x));
    }

    if let Some(cmd) = opt.cmd {
        notifier.set_title(Title::from(&cmd));

        match cmd {
            Command::Backlight => {
                if let Some(i) = config.get_icon() {
                    icon.set_theme(i.get_theme());
                }

                if let Some(bac) = config.get_backlight() {
                    notifier.set_title(Title::Custom(
                        bac.get_title()
                            .unwrap_or_else(|| Title::Backlight.to_string()),
                    ));
                }

                notifier.set_body(read::backlight(&mut icon))
            }
            Command::Battery => {
                if let Some(i) = config.get_icon() {
                    icon.set_theme(i.get_theme());
                }

                if let Some(bat) = config.get_battery() {
                    notifier.set_title(Title::Custom(
                        bat.get_title()
                            .unwrap_or_else(|| Title::Battery.to_string()),
                    ));
                }

                notifier.set_body(read::battery(&config, &mut icon))
            }
            Command::Uptime => {
                if let Some(i) = config.get_icon() {
                    icon.set_theme(i.get_theme());
                }

                if let Some(up) = config.get_uptime() {
                    notifier.set_title(Title::Custom(
                        up.get_title().unwrap_or_else(|| Title::Uptime.to_string()),
                    ));
                }

                notifier.set_body(read::uptime(&mut icon))
            }
            Command::Network => {
                if let Some(i) = config.get_icon() {
                    icon.set_theme(i.get_theme());
                }

                if let Some(net) = config.get_network() {
                    notifier.set_title(Title::Custom(
                        net.get_title()
                            .unwrap_or_else(|| Title::Network.to_string()),
                    ));
                }

                notifier.set_body(read::network(&config, &mut icon))
            }
            Command::Date => {
                if let Some(i) = config.get_icon() {
                    icon.set_theme(i.get_theme());
                }

                if let Some(date) = config.get_date() {
                    notifier.set_title(Title::Custom(
                        date.get_title().unwrap_or_else(|| Title::Date.to_string()),
                    ));
                }

                notifier.set_body(read::date(&config, &mut icon))
            }
            Command::Memory => {
                if let Some(i) = config.get_icon() {
                    icon.set_theme(i.get_theme());
                }

                if let Some(mem) = config.get_memory() {
                    notifier.set_title(Title::Custom(
                        mem.get_title().unwrap_or_else(|| Title::Memory.to_string()),
                    ));
                }

                notifier.set_body(read::memory(&mut icon))
            }
        }

        notifier.set_timeout(config);
        notifier.set_icon(icon);
        notifier.send();
    }
}
