use crate::options::Command;
use crate::Config;
use crate::Icon;

use libmacchina::traits::ReadoutError;
use notify_rust::Notification;
use notify_rust::Timeout;

use std::fmt;

pub struct Notifier {
    /// The header of a notification.
    title: Title,
    /// The body of a notification.
    body: String,
    /// The timeout of a notification.
    timeout: u32,
    /// A path to an icon as a string.
    icon: String,
}

impl Notifier {
    pub fn new() -> Notifier {
        Notifier {
            title: Title::Date,
            timeout: 5,
            icon: String::new(),
            body: String::new(),
        }
    }

    pub fn set_title(&mut self, title: Title) {
        self.title = title;
    }

    pub fn get_title(&self) -> String {
        self.title.to_string()
    }

    pub fn set_icon(&mut self, icon: Icon) {
        let path = icon.get_path();

        if !path.exists() {
            return;
        }

        if let Some(str) = path.to_str() {
            self.icon = str.to_string();
        }
    }

    pub fn set_timeout(&mut self, config: Config) {
        if let Some(timeout) = config.get_timeout() {
            self.timeout = timeout;
        }
    }

    pub fn get_timeout(&self) -> Timeout {
        Timeout::Milliseconds(self.timeout * 1000)
    }

    pub fn set_body(&mut self, body: Result<String, ReadoutError>) {
        match body {
            Ok(b) => self.body = b,
            Err(e) => self.body = e.to_string(),
        }
    }

    pub fn get_body(&self) -> String {
        self.body.to_owned()
    }

    pub fn send(&self) {
        Notification::new()
            .body(&self.get_body())
            .summary(&self.get_title())
            .timeout(self.get_timeout())
            .icon(&self.icon)
            .show()
            .expect("Verify that a notification daemon is running in the background");
    }
}

#[derive(Debug, PartialEq, Clone)]
pub enum Title {
    Date,
    Uptime,
    Memory,
    Battery,
    Network,
    Backlight,
    Custom(String),
}

impl fmt::Display for Title {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match &*self {
            Title::Date => write!(f, "Date"),
            Title::Uptime => write!(f, "Uptime"),
            Title::Memory => write!(f, "Memory"),
            Title::Battery => write!(f, "Battery"),
            Title::Network => write!(f, "Network"),
            Title::Backlight => write!(f, "Backlight"),
            Title::Custom(s) => write!(f, "{}", s),
        }
    }
}

impl From<&Command> for Title {
    fn from(cmd: &Command) -> Self {
        match cmd {
            Command::Date => Title::Date,
            Command::Uptime => Title::Uptime,
            Command::Memory => Title::Memory,
            Command::Battery => Title::Battery,
            Command::Network => Title::Network,
            Command::Backlight => Title::Backlight,
        }
    }
}
