use linicon::lookup_icon;
use std::path::PathBuf;

pub struct Icon {
    /// A path to an icon.
    path: PathBuf,
    /// The name of an icon theme.
    theme: String,
    /// The base location of the `icons` directory.
    location: PathBuf,
}

impl Icon {
    pub fn new() -> Self {
        let base = if cfg!(target_os = "linux") {
            PathBuf::from("/usr/share/icons")
        } else if cfg!(target_os = "netbsd") {
            PathBuf::from("/usr/pkg/share/icons")
        } else {
            PathBuf::new()
        };

        Icon {
            path: PathBuf::new(),
            theme: String::new(),
            location: base,
        }
    }

    pub fn get_location(&self) -> &str {
        self.location.to_str().unwrap_or_default()
    }

    pub fn get_path(&self) -> PathBuf {
        self.path.to_owned()
    }

    pub fn get_available_themes(&self) -> Vec<String> {
        let mut themes = vec![];
        let mut places = vec![self.location.clone()];

        if let Ok(home) = std::env::var("HOME") {
            places.push(PathBuf::from(home).join(".icons"));
        }

        places.iter().for_each(|place| {
            if let Ok(entries) = place.read_dir() {
                for entry in entries.flatten() {
                    if !entry.path().join("index.theme").exists() {
                        continue;
                    }

                    if let Some(name) = entry.path().file_name() {
                        themes.push(name.to_string_lossy().to_string());
                    }
                }
            }
        });

        themes
    }

    pub fn set_theme(&mut self, theme: Option<String>) {
        if let Some(t) = theme {
            self.theme = t;
        }
    }

    pub fn find_icon(&mut self, icon_name: &str) {
        if self.theme.is_empty() {
            return;
        }

        if let Ok(icon) = lookup_icon(icon_name).with_search_paths(&[self.get_location()]) {
            if let Some(icon) = icon.from_theme(&self.theme).next() {
                match icon {
                    Ok(i) => self.path = i.path,
                    _ => {
                        // The requested icon theme does not exist.
                        eprintln!("Warning: \"{}\" doesn't exist.", &self.theme);
                        // Print a list of usable themes.
                        eprintln!("Pick a theme from this list:");
                        self.get_available_themes()
                            .into_iter()
                            .filter(|t| !t.eq("default"))
                            .for_each(|t| eprintln!(" > {}", t));
                    }
                }

                return;
            }

            // The requested icon does not exist for the given theme.
            eprintln!(
                "Warning: \"{}\" doesn't exist for \"{}\".",
                icon_name, &self.theme
            )
        }
    }
}
