use std::fmt;
use std::str::FromStr;
use structopt::clap::AppSettings;
use structopt::StructOpt;

#[derive(Debug)]
pub enum CommandError {
    Unknown,
}

impl fmt::Display for CommandError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            CommandError::Unknown => write!(f, "Unknown flag."),
        }
    }
}

#[derive(StructOpt, Debug, Clone)]
pub enum Command {
    #[structopt(visible_alias = "B", about = "Show a backlight notification", settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion])]
    Backlight,
    #[structopt(visible_alias = "b", about = "Show a backlight notification", settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion])]
    Battery,
    #[structopt(visible_alias = "n", about = "Show a backlight notification", settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion])]
    Network,
    #[structopt(visible_alias = "u", about = "Show a backlight notification", settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion])]
    Uptime,
    #[structopt(visible_alias = "m", about = "Show a backlight notification", settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion])]
    Memory,
    #[structopt(visible_alias = "d", about = "Show a backlight notification", settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion])]
    Date,
}

impl fmt::Display for Command {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Command::Backlight => write!(f, "Backlight"),
            Command::Battery => write!(f, "Battery"),
            Command::Network => write!(f, "Network"),
            Command::Uptime => write!(f, "Uptime"),
            Command::Memory => write!(f, "Memory"),
            Command::Date => write!(f, "Date"),
        }
    }
}

impl FromStr for Command {
    type Err = CommandError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match &s.to_lowercase()[..] {
            "backlight" => Ok(Command::Backlight),
            "battery" => Ok(Command::Battery),
            "network" => Ok(Command::Network),
            "uptime" => Ok(Command::Uptime),
            "memory" => Ok(Command::Memory),
            "date" => Ok(Command::Date),
            _ => Err(CommandError::Unknown),
        }
    }
}
#[derive(Debug, StructOpt, Default)]
#[structopt(settings = &[AppSettings::ArgsNegateSubcommands, AppSettings::ArgRequiredElseHelp, AppSettings::DisableHelpSubcommand])]
pub struct Options {
    /// Notifies you about your system's uptime.
    #[structopt(subcommand, help = "Send a notification with uptime information")]
    pub cmd: Option<Command>,

    /// Lists the available icon themes.
    #[structopt(short = "L", help = "Returns a list of the available icon themes")]
    pub icon_themes: bool,

    /// Provides debug information.
    #[structopt(long = "debug", short = "D", help = "Provides debug information")]
    pub debug: bool,
}
