use super::Iconize;
use crate::Icon;
use bytesize::ByteSize;
use std::fmt;

pub struct NetworkFormatter {
    logical_address: Option<String>,
    rx_bytes: Option<usize>,
    tx_bytes: Option<usize>,
}

impl NetworkFormatter {
    pub fn new() -> Self {
        Self {
            logical_address: None,
            rx_bytes: None,
            tx_bytes: None,
        }
    }

    pub fn set_logical_address(&mut self, addr: String) {
        self.logical_address = Some(addr)
    }

    pub fn get_logical_address(&self) -> Option<String> {
        self.logical_address.to_owned()
    }

    pub fn set_rx_bytes(&mut self, bytes: usize) {
        self.rx_bytes = Some(bytes);
    }

    pub fn set_tx_bytes(&mut self, bytes: usize) {
        self.tx_bytes = Some(bytes);
    }

    pub fn get_rx_bytes(&self) -> Option<ByteSize> {
        self.rx_bytes.map(|bytes| ByteSize::kb(bytes as u64 / 1024))
    }

    pub fn get_tx_bytes(&self) -> Option<ByteSize> {
        self.tx_bytes.map(|bytes| ByteSize::kb(bytes as u64 / 1024))
    }
}

impl Iconize for NetworkFormatter {
    fn set_icon(&self, icon: &mut Icon) {
        match self.get_logical_address() {
            Some(_) => icon.find_icon("network-wireless-symbolic"),
            _ => icon.find_icon("network-wireless-disconnected-symbolic"),
        }
    }
}

impl fmt::Display for NetworkFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if let Some(addr) = &self.get_logical_address() {
            writeln!(f, "Local IP: {}", addr);
        }

        match (self.get_rx_bytes(), self.get_tx_bytes()) {
            (Some(rx), Some(tx)) => write!(f, "Data: {} 🠗 / {} 🠕", rx, tx),
            _ => Ok(()),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display_network() {
        let mut fmt = NetworkFormatter::new();

        fmt.set_logical_address("127.0.0.1".to_owned());
        assert_eq!(fmt.to_string(), "Local IP: 127.0.0.1\n".to_owned());

        fmt.set_logical_address("192.168.1.1".to_owned());
        fmt.set_rx_bytes(717000);
        fmt.set_tx_bytes(717000);
        assert_eq!(
            fmt.to_string(),
            "Local IP: 192.168.1.1\nData: 700.0 KB 🠗 / 700.0 KB 🠕".to_owned()
        );
    }
}
