use crate::fmt::date::{Day, Meridiem, Month};
use crate::fmt::Iconize;
use crate::Icon;
use chrono::Weekday;
use std::fmt;

#[derive(Debug, PartialEq)]
pub struct DateFormatter {
    hour: u32,
    minute: u32,
    day: u32,
    month: Month,
    railway: bool,
    weekday: Day,
    meridiem: Meridiem,
    format: String,
}

impl DateFormatter {
    pub fn new() -> Self {
        Self {
            hour: 0,
            minute: 0,
            day: 0,
            month: Month::January,
            railway: false,
            meridiem: Meridiem::Ante,
            weekday: Day::Monday,
            format: String::new(),
        }
    }

    pub fn from_config(&mut self, format: String) {
        self.format = format
            .replace("%HH", &format!("{:0>2}", &self.hour))
            .replace("%MM", &format!("{:0>2}", &self.minute))
            .replace("%P", &self.meridiem.to_string())
            .replace("%w", &self.weekday.short().to_string())
            .replace("%W", &self.weekday.long().to_string())
            .replace("%m", &self.month.short().to_string())
            .replace("%M", &self.month.long().to_string())
            .replace("%d", &self.day.to_string())
    }

    pub fn set_hour(&mut self, hour: u32) {
        self.hour = hour;
    }

    #[cfg(test)]
    pub fn use_railway(&mut self) {
        self.railway = true;
    }

    pub fn set_minute(&mut self, minute: u32) {
        self.minute = minute;
    }

    pub fn set_day(&mut self, day: u32) {
        self.day = day;
    }

    pub fn set_month(&mut self, month: u32) {
        self.month = month.into();
    }

    pub fn set_weekday(&mut self, weekday: Weekday) {
        self.weekday = weekday.into();
    }

    pub fn set_meridiem(&mut self, pm: bool) {
        if pm {
            self.meridiem = Meridiem::Post;
        }
    }
}

impl Iconize for DateFormatter {
    fn set_icon(&self, icon: &mut Icon) {
        let night = if self.railway {
            self.hour >= 20 || self.hour <= 6
        } else {
            self.hour >= 8 && self.meridiem == Meridiem::Post
                || self.hour <= 6 && self.meridiem == Meridiem::Ante
        };

        if night {
            icon.find_icon("weather-clear-night-symbolic")
        } else {
            icon.find_icon("weather-clear-symbolic")
        }
    }
}

impl fmt::Display for DateFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if !self.format.is_empty() {
            return write!(f, "{}", self.format);
        }

        if self.railway {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02}",
                self.weekday.short(),
                self.month.short(),
                self.day,
                self.hour,
                self.minute,
            )
        } else {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02} {}",
                self.weekday.short(),
                self.month.short(),
                self.day,
                self.hour,
                self.minute,
                self.meridiem,
            )
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    #[test]
    fn from_u32_to_month() {
        let january = Month::from(1);
        assert_eq!(january, Month::January);
    }

    #[test]
    fn display_month_short() {
        let february = Month::from(2).short().to_string();
        assert_eq!(february, "Feb".to_owned());
    }

    #[test]
    fn display_month_long() {
        let february = Month::from(2).long().to_string();
        assert_eq!(february, "February".to_owned());
    }

    #[test]
    fn display_date_railway() {
        let mut fmt = DateFormatter::new();
        fmt.use_railway();
        fmt.set_weekday(Weekday::Mon);
        fmt.set_day(13);
        fmt.set_month(9);
        fmt.set_hour(16);
        fmt.set_minute(12);
        assert_eq!(fmt.to_string(), "Mon, Sep 13 at 16:12".to_owned());
    }

    #[test]
    fn display_date() {
        let mut fmt = DateFormatter::new();
        fmt.set_weekday(Weekday::Sat);
        fmt.set_meridiem(true);
        fmt.set_day(1);
        fmt.set_month(8);
        fmt.set_hour(2);
        fmt.set_minute(37);
        assert_eq!(fmt.to_string(), "Sat, Aug 01 at 02:37 PM".to_owned());
    }

    #[test]
    fn date_from_config() {
        let mut fmt = DateFormatter::new();
        fmt.from_config("%w".to_string());
        assert_eq!(fmt.to_string(), "Mon".to_string());
        fmt.from_config("%W".to_string());
        assert_eq!(fmt.to_string(), "Monday".to_string());
        fmt.from_config("%m".to_string());
        assert_eq!(fmt.to_string(), "Jan".to_string());
        fmt.from_config("%M".to_string());
        assert_eq!(fmt.to_string(), "January".to_string());
    }
}
