//! This module is responsible for reading data from
//! [libmacchina](https://docs.rs/libmacchina/latest/libmacchina/) and sending it to its respective
//! formatter.

use libmacchina::traits::BatteryReadout as _;
use libmacchina::traits::GeneralReadout as _;
use libmacchina::traits::MemoryReadout as _;
use libmacchina::traits::NetworkReadout as _;
use libmacchina::traits::ReadoutError;
use libmacchina::{BatteryReadout, GeneralReadout, MemoryReadout, NetworkReadout};

use crate::config::Config;
use crate::fmt::backlight::BacklightFormatter;
use crate::fmt::battery::BatteryFormatter;
use crate::fmt::date::DateFormatter;
use crate::fmt::memory::MemoryFormatter;
use crate::fmt::network::NetworkFormatter;
use crate::fmt::uptime::UptimeFormatter;
use crate::fmt::Iconize;
use crate::helpers;
use crate::Icon;

use chrono::{Datelike, Timelike};

pub fn uptime(icon: &mut Icon) -> Result<String, ReadoutError> {
    let readout = GeneralReadout::new();
    let fmt = UptimeFormatter::from(readout.uptime()?);
    fmt.set_icon(icon);

    Ok(fmt.to_string())
}

pub fn network(config: Option<&Config>, icon: &mut Icon) -> Result<String, ReadoutError> {
    let mut fmt = NetworkFormatter::new();
    let readout = NetworkReadout::new();

    if let Some(conf) = config {
        if let Some(network) = conf.get_network() {
            if let Some(interface) = network.get_interface() {
                let ifname = Some(interface.as_str());
                fmt.set_logical_address(readout.logical_address(ifname)?);

                if let Ok(rx_bytes) = readout.rx_bytes(ifname) {
                    fmt.set_rx_bytes(rx_bytes);
                }

                if let Ok(tx_bytes) = readout.tx_bytes(ifname) {
                    fmt.set_tx_bytes(tx_bytes);
                }
            }
        }
    }

    fmt.set_icon(icon);

    Ok(fmt.to_string())
}

pub fn memory(icon: &mut Icon) -> Result<String, ReadoutError> {
    let mut fmt = MemoryFormatter::new();
    let readout = MemoryReadout::new();

    fmt.set_total(readout.total()?);
    fmt.set_used(readout.used()?);
    fmt.set_percentage();
    fmt.set_icon(icon);

    Ok(fmt.to_string())
}

pub fn battery(icon: &mut Icon) -> Result<String, ReadoutError> {
    let mut fmt = BatteryFormatter::new();
    let readout = BatteryReadout::new();

    fmt.set_percentage(readout.percentage()?);
    fmt.set_state(readout.status()?);
    fmt.set_icon(icon);

    Ok(fmt.to_string())
}

pub fn backlight(icon: &mut Icon) -> Result<String, ReadoutError> {
    let mut fmt = BacklightFormatter::new();
    let readout = GeneralReadout::new();

    fmt.set_percentage(readout.backlight()?);
    fmt.set_icon(icon);

    Ok(fmt.to_string())
}

pub fn date(config: Option<&Config>, icon: &mut Icon) -> Result<String, ReadoutError> {
    let mut fmt = DateFormatter::new();
    if let Some(readout) = helpers::utc_from_timezone() {
        let hour = readout.hour12();
        fmt.set_weekday(readout.weekday());
        fmt.set_month(readout.month());
        fmt.set_day(readout.day());
        fmt.set_minute(readout.minute());
        fmt.set_meridiem(hour.0);
        fmt.set_hour(hour.1);

        if let Some(conf) = config {
            if let Some(date) = conf.get_date() {
                if date.is_railway() {
                    let hour = readout.hour();
                    fmt.set_hour(hour);
                } else {
                    let hour = readout.hour12();
                    fmt.set_meridiem(hour.0);
                    fmt.set_hour(hour.1);
                }

                if let Some(format) = date.get_format() {
                    fmt.from_config(format);
                }
            }
        }

        fmt.set_icon(icon);

        return Ok(fmt.to_string());
    }

    Err(ReadoutError::Other(
        "I'm sorry! I couldn't retrieve your timezone.".to_string(),
    ))
}
