#![allow(unused_must_use)]

// The flow of citron is as follows:
//
//   read   ---->   format   ---->   notify
//    |              |                |
//    |              |                |
//    |              |                `---> `notifier` module
//    |              |
//    |              `---> `fmt` modules
//    |
//    `--> `read` module

mod config;
mod debug;
mod fmt;
mod helpers;
mod icon;
mod notifier;
mod options;
mod read;

use config::Config;
use debug::get_debug;
use icon::Icon;
use notifier::{Notifier, Summary};
use options::Command;
use options::Options;
use owo_colors::OwoColorize;
use structopt::StructOpt;

fn main() {
    let mut notifier = Notifier::new();
    let mut icon = Icon::new();
    let opt = Options::from_args();
    let config = Config::read_config();

    if opt.debug {
        get_debug();
        return;
    }

    if opt.icon_themes {
        println!("{}", "Available icon themes:".bold());
        icon.get_available_themes()
            .into_iter()
            .filter(|x| !x.eq("default"))
            .for_each(|x| eprintln!(" {} {}", ">".blue(), x));
    }

    if let Some(cmd) = opt.cmd {
        notifier.set_summary(Summary::from(&cmd));

        match cmd {
            Command::Backlight => {
                icon.set_theme(config.as_ref());
                notifier.set_body(read::backlight(&mut icon));
                if let Some(conf) = config.as_ref() {
                    if let Some(bac) = conf.get_backlight() {
                        notifier.set_summary(Summary::Custom(
                            bac.get_title().unwrap_or_else(|| Summary::Date.to_string()),
                        ));
                    }
                }
            }
            Command::Battery => {
                icon.set_theme(config.as_ref());
                notifier.set_body(read::battery(&mut icon));
                if let Some(conf) = config.as_ref() {
                    if let Some(bat) = conf.get_battery() {
                        notifier.set_summary(Summary::Custom(
                            bat.get_title().unwrap_or_else(|| Summary::Date.to_string()),
                        ));
                    }
                }
            }
            Command::Uptime => {
                icon.set_theme(config.as_ref());
                notifier.set_body(read::uptime(&mut icon));
                if let Some(conf) = config.as_ref() {
                    if let Some(up) = conf.get_uptime() {
                        notifier.set_summary(Summary::Custom(
                            up.get_title().unwrap_or_else(|| Summary::Date.to_string()),
                        ));
                    }
                }
            }
            Command::Network => {
                icon.set_theme(config.as_ref());
                notifier.set_body(read::network(config.as_ref(), &mut icon));
                if let Some(conf) = config.as_ref() {
                    if let Some(net) = conf.get_network() {
                        notifier.set_summary(Summary::Custom(
                            net.get_title()
                                .unwrap_or_else(|| Summary::Network.to_string()),
                        ));
                    }
                }
            }
            Command::Date => {
                icon.set_theme(config.as_ref());
                notifier.set_body(read::date(config.as_ref(), &mut icon));
                if let Some(conf) = config.as_ref() {
                    if let Some(date) = conf.get_date() {
                        notifier.set_summary(Summary::Custom(
                            date.get_title()
                                .unwrap_or_else(|| Summary::Date.to_string()),
                        ));
                    }
                }
            }
            Command::Memory => {
                icon.set_theme(config.as_ref());
                notifier.set_body(read::memory(&mut icon));
                if let Some(conf) = config.as_ref() {
                    if let Some(mem) = conf.get_memory() {
                        notifier.set_summary(Summary::Custom(
                            mem.get_title().unwrap_or_else(|| Summary::Date.to_string()),
                        ));
                    }
                }
            }
        }

        notifier.set_timeout(config.as_ref());
        notifier.set_icon(&icon);
        notifier.send();
    }
}
