use super::Iconize;
use crate::Icon;
use chrono::Weekday;
use std::fmt;

#[derive(Debug, PartialEq)]
pub struct DateFormatter {
    hour: u32,
    minute: u32,
    day: u32,
    month: u32,
    railway: bool,
    weekday: Weekday,
    meridiem: Meridiem,
    format: String,
}

impl DateFormatter {
    pub fn new() -> Self {
        Self {
            hour: 0,
            minute: 0,
            day: 0,
            month: 0,
            railway: false,
            meridiem: Meridiem::Ante,
            weekday: Weekday::Mon,
            format: String::new(),
        }
    }

    pub fn from_config(&mut self, format: String) {
        self.format = format
            .replace("%HH", &format!("{:0>2}", &self.hour))
            .replace("%MM", &format!("{:0>2}", &self.minute))
            .replace("%P", &self.meridiem.to_string())
            .replace("%w", &self.weekday.to_string())
            .replace("%m", &Month::from(self.month).short().to_string())
            .replace("%M", &Month::from(self.month).long().to_string())
            .replace("%d", &self.day.to_string())
    }

    pub fn set_hour(&mut self, hh: u32) {
        self.hour = hh;
    }

    pub fn set_minute(&mut self, mm: u32) {
        self.minute = mm;
    }

    pub fn set_day(&mut self, dd: u32) {
        self.day = dd;
    }

    pub fn set_month(&mut self, mm: u32) {
        self.month = mm;
    }

    pub fn set_weekday(&mut self, wk: Weekday) {
        self.weekday = wk;
    }

    pub fn set_meridiem(&mut self, pm: bool) {
        if pm {
            self.meridiem = Meridiem::Post;
        }
    }
}

impl Iconize for DateFormatter {
    fn set_icon(&self, icon: &mut Icon) {
        let night = if self.railway {
            self.hour >= 20 || self.hour <= 6
        } else {
            self.hour >= 8 && self.meridiem == Meridiem::Post
                || self.hour <= 6 && self.meridiem == Meridiem::Ante
        };

        if night {
            icon.find_icon("weather-clear-night-symbolic")
        } else {
            icon.find_icon("weather-clear-symbolic")
        }
    }
}

impl fmt::Display for DateFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if !self.format.is_empty() {
            return write!(f, "{}", self.format);
        }

        if self.railway {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02}",
                self.weekday,
                Month::from(self.month).short(),
                self.day,
                self.hour,
                self.minute,
            )
        } else {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02} {}",
                self.weekday,
                Month::from(self.month).short(),
                self.day,
                self.hour,
                self.minute,
                self.meridiem,
            )
        }
    }
}

#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Meridiem {
    Ante,
    Post,
}

impl fmt::Display for Meridiem {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Meridiem::Ante => write!(f, "AM"),
            Meridiem::Post => write!(f, "PM"),
        }
    }
}

#[derive(Debug, PartialEq)]
pub enum Month {
    January,
    February,
    March,
    April,
    May,
    June,
    July,
    August,
    September,
    October,
    November,
    December,
}

impl From<u32> for Month {
    fn from(s: u32) -> Month {
        match s {
            1 => Month::January,
            2 => Month::February,
            3 => Month::March,
            4 => Month::April,
            5 => Month::May,
            6 => Month::June,
            7 => Month::July,
            8 => Month::August,
            9 => Month::September,
            10 => Month::October,
            11 => Month::November,
            _ => Month::December,
        }
    }
}

impl Month {
    fn short<'a>(&'a self) -> impl fmt::Display + 'a {
        struct Short<'a>(&'a Month);
        impl<'a> fmt::Display for Short<'a> {
            fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
                match self.0 {
                    Month::January => write!(f, "Jan"),
                    Month::February => write!(f, "Feb"),
                    Month::March => write!(f, "Mar"),
                    Month::April => write!(f, "Apr"),
                    Month::May => write!(f, "May"),
                    Month::June => write!(f, "Jun"),
                    Month::July => write!(f, "Jul"),
                    Month::August => write!(f, "Aug"),
                    Month::September => write!(f, "Sep"),
                    Month::October => write!(f, "Oct"),
                    Month::November => write!(f, "Nov"),
                    Month::December => write!(f, "Dec"),
                }
            }
        }
        Short(self)
    }

    fn long<'a>(&'a self) -> impl fmt::Display + 'a {
        struct Long<'a>(&'a Month);
        impl<'a> fmt::Display for Long<'a> {
            fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
                match self.0 {
                    Month::January => write!(f, "January"),
                    Month::February => write!(f, "February"),
                    Month::March => write!(f, "March"),
                    Month::April => write!(f, "April"),
                    Month::May => write!(f, "May"),
                    Month::June => write!(f, "June"),
                    Month::July => write!(f, "July"),
                    Month::August => write!(f, "August"),
                    Month::September => write!(f, "September"),
                    Month::October => write!(f, "October"),
                    Month::November => write!(f, "November"),
                    Month::December => write!(f, "December"),
                }
            }
        }
        Long(self)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn from_u32_to_month() {
        let january = Month::from(1);
        assert_eq!(january, Month::January);
    }

    #[test]
    fn display_month_short() {
        let february = Month::from(2).short().to_string();
        assert_eq!(february, "Feb".to_owned());
    }

    #[test]
    fn display_month_long() {
        let february = Month::from(2).long().to_string();
        assert_eq!(february, "February".to_owned());
    }

    #[test]
    fn display_date_railway() {
        let mut fmt = DateFormatter::new();
        fmt.railway = true;
        fmt.set_weekday(Weekday::Mon);
        fmt.set_day(13);
        fmt.set_month(9);
        fmt.set_hour(16);
        fmt.set_minute(12);
        assert_eq!(fmt.to_string(), "Mon, Sep 13 at 16:12".to_owned());
    }

    #[test]
    fn display_date() {
        let mut fmt = DateFormatter::new();
        fmt.set_weekday(Weekday::Sat);
        fmt.set_meridiem(true);
        fmt.set_day(1);
        fmt.set_month(8);
        fmt.set_hour(2);
        fmt.set_minute(37);
        assert_eq!(fmt.to_string(), "Sat, Aug 01 at 02:37 PM".to_owned());
    }
}
