use structopt::StructOpt;

#[derive(Debug, StructOpt)]
pub struct Opt {
    /// Sets the timeout for all the notifiers.
    #[structopt(long = "timeout", short = "t")]
    timeout: u32,

    /// Notifies you about your system's uptime.
    #[structopt(long = "uptime", short = "u")]
    uptime: bool,

    /// Notifies you about your system's date and time.
    #[structopt(long = "date-time", short = "d")]
    date_time: bool,

    /// Notifies you about your system's battery percentage and status.
    #[structopt(long = "battery", short = "b")]
    battery: bool,

    /// Notifies you about your system's networking status.
    #[structopt(long = "network", short = "n")]
    network: bool,

    /// Notifies you about your system's memory usage.
    #[structopt(long = "memory", short = "m")]
    memory: bool,

    /// Specify an interface for the network notifier.
    #[structopt(long = "interface", short = "i")]
    interface: Option<String>,

}

impl Default for Opt {
    fn default() -> Self {
        Opt {
            timeout: 5,
            uptime: false,
            date_time: false,
            battery: false,
            network: false,
            memory: false,
            interface: None,
        }
    }
}

impl Opt {
    pub fn get_timeout(&self) -> u32 {
        self.timeout * 1000
    }

    pub fn is_requesting_date_time(&self) -> bool {
        self.date_time
    }

    pub fn is_requesting_uptime(&self) -> bool {
        self.uptime
    }

    pub fn is_requesting_battery(&self) -> bool {
        self.battery
    }

    pub fn is_requesting_network(&self) -> bool {
        self.network
    }

    pub fn is_requesting_memory(&self) -> bool {
        self.memory
    }

    pub fn get_interface(&self) -> Option<String> {
        self.interface.to_owned()
    }
}
