use cita_vm::json_tests::common::*;
use cita_vm::*;
use state::State;
use std::fs;
use std::io;
use std::io::Write;
use std::sync::Arc;
use std::thread;

fn test_json_file(p: &str) {
    let f = fs::File::open(p).unwrap();
    let t = cita_vm::json_tests::general_state_test::Test::load(f).unwrap();
    for (name, data) in t.into_iter() {
        let data_post_homestead = data.post.unwrap().homestead;
        if data_post_homestead.is_none() {
            continue;
        }

        for (i, postdata) in data_post_homestead.unwrap().iter().enumerate() {
            io::stderr()
                .write_all(format!("{}::{}::{}\n", p, name, i).as_bytes())
                .unwrap();
            let d = Arc::new(cita_trie::MemoryDB::new(false));
            let mut state_provider = State::new(d).unwrap();

            for (address, account) in data.pre.clone().unwrap() {
                let balance = string_2_u256(account.balance);
                let code = string_2_bytes(account.code);
                let nonce = string_2_u256(account.nonce);
                if code.is_empty() {
                    state_provider.new_contract(&address, balance, nonce, vec![]);
                } else {
                    state_provider.new_contract(&address, balance, nonce, code);
                }
                for (k, v) in account.storage {
                    let kk = string_2_h256(k);
                    let vv = string_2_h256(v);
                    state_provider.set_storage(&address, kk, vv).unwrap();
                }
            }
            state_provider.commit().unwrap();

            let idx_gas = &postdata.indexes[&String::from("gas")];
            let idx_value = &postdata.indexes[&String::from("value")];
            let idx_data = &postdata.indexes[&String::from("data")];

            let str_block_gas = data.env.current_gas_limit.clone();
            let str_gas = data.transaction.gas_limit.clone()[*idx_gas].clone();
            let str_value = data.transaction.value.clone()[*idx_value].clone();
            let str_data = data.transaction.data.clone()[*idx_data].clone();

            let evm_context = evm::Context {
                gas_limit: string_2_u256(str_block_gas.clone()).low_u64(),
                coinbase: data.env.current_coinbase,
                number: string_2_u256(data.env.current_number.clone()),
                timestamp: string_2_u256(data.env.current_timestamp.clone()).low_u64(),
                difficulty: string_2_u256(data.env.current_difficulty.clone()),
            };
            let mut cfg = Config::default();
            cfg.block_gas_limit = string_2_u256(data.env.current_gas_limit.clone()).low_u64();
            let mut tx = Transaction {
                from: secret_2_address(data.transaction.secret_key.as_str()),
                to: None,
                value: string_2_u256(str_value),
                nonce: string_2_u256(data.transaction.nonce.clone()),
                gas_limit: string_2_u256(str_gas).low_u64(),
                gas_price: string_2_u256(data.transaction.gas_price.clone()),
                input: string_2_bytes(str_data),
            };
            if !data.transaction.to.is_empty() {
                tx.to = Some(string_2_address(data.transaction.to.clone()));
            }
            let exepinst = Executive::new(Arc::new(BlockDataProviderMock::default()), state_provider, cfg);
            let _exec_result = exepinst.exec(evm_context, tx);
            let root = exepinst.commit().unwrap();
            // let _ = exec(
            //     Arc::new(Box::new(BlockDataProviderMock::default())),
            //     state_provider.clone(),
            //     evm_context,
            //     cfg,
            //     tx,
            // );
            // let root = state_provider.borrow_mut().root;
            assert_eq!(root, string_2_h256(postdata.hash.clone()));
        }
    }
}

#[allow(dead_code)]
fn test_json_path_skip(p: &str, skip: Vec<String>) {
    let info = fs::metadata(p).unwrap();
    if info.is_dir() {
        for entry in fs::read_dir(p).unwrap() {
            let entry = entry.unwrap();
            let p = entry.path();
            if skip.contains(&entry.file_name().into_string().unwrap()) {
                continue;
            }
            test_json_path(p.to_str().unwrap());
        }
    } else {
        test_json_file(p);
    }
}

#[allow(dead_code)]
fn test_json_path(p: &str) {
    test_json_path_skip(p, vec![])
}

// The reason contains "bug", "overflow", "slow", "precompile" and "obsolete".
#[allow(dead_code)]
fn skip(_reason: &str, _name: &str) {}

#[rustfmt::skip]
#[test]
fn test_state_work() {
    let _ = env_logger::builder().is_test(true).try_init();
}

#[rustfmt::skip]
#[test]
fn test_state_pass() {
    thread::Builder::new().stack_size(134_217_728).spawn(move || {
        /*test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/addmodNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/addNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/andNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/balanceNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/byteNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/callcodeNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/calldatacopyNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/calldataloadNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/callNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/codecopyNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/createNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/delegatecallNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/divNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/eqNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/expNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/extcodecopyNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/extcodesizeNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/gtNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/iszeroNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/jumpiNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/jumpNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/log0NonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/log1NonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/log2NonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/log3NonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/ltNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/mloadNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/modNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/mstore8NonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/mstoreNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/mulmodNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/mulNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/notNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/orNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/returnNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/sdivNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/sgtNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/sha3NonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/signextNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/sloadNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/sltNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/smodNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/sstoreNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/subNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/suicideNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stArgsZeroOneBalance/xorNonConst.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stAttackTest/ContractCreationSpam.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stAttackTest/CrashingTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stBadOpcode/badOpcodes.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stBugs/evmBytecode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stBugs/returndatacopyPythonBug_Tue_03_48_41-1432.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stBugs/staticcall_createfails.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcallcode_001.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcallcode_001_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcallcode_001_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcallcode_001_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcallcode_001_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcallcode_001_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcall_000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcall_000_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcall_000_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcall_000_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcall_000_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcall_000_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecallcode_011.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecallcode_011_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecallcode_011_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecallcode_011_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecallcode_011_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecallcode_011_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecall_010.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecall_010_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecall_010_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecall_010_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecall_010_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecall_010_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcode_01.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcode_01_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcallcode_01_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcall_00.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcall_00_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcall_00_OOGE_valueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcall_00_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcallcode_101.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcallcode_101_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcallcode_101_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcallcode_101_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcallcode_101_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcallcode_101_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcall_100.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcall_100_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcall_100_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcall_100_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcall_100_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcall_100_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecallcode_111.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecallcode_111_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecallcode_111_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecallcode_111_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecallcode_111_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecallcode_111_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecall_110.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecall_110_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecall_110_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecall_110_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecall_110_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecall_110_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcode_11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcode_11_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecallcode_11_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecall_10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecall_10_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodecall_10_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodeDynamicCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodeDynamicCode2SelfCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodeEmptycontract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodeInInitcodeToEmptyContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodeInInitcodeToExisContractWithVTransferNEMoney.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodeInInitcodeToExistingContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcodeInInitcodeToExistingContractWithValueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/callcode_checkPC.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/call_OOG_additionalGasCosts1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCodes/call_OOG_additionalGasCosts2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/Call1024BalanceTooLow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/Call1024OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/Call1024PreCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/Callcode1024BalanceTooLow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/Callcode1024OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/CallcodeLoseGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeOutput1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeOutput2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeOutput3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeOutput3Fail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeOutput3partial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeOutput3partialFail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeWithHighValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callcodeWithHighValueAndGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/CallLoseGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callOutput1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callOutput2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callOutput3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callOutput3Fail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callOutput3partial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callOutput3partialFail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/CallRecursiveBombPreCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callWithHighValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callWithHighValueAndGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callWithHighValueAndOOGatTxLevel.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/callWithHighValueOOGinCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createFailBalanceTooLow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createInitFailBadJumpDestination.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createInitFailStackSizeLargerThan1024.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createInitFailStackUnderflow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createInitFailUndefinedInstruction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createInitFail_OOGduringInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createInitOOGforCREATE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createJS_ExampleContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createJS_NoCollision.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createNameRegistratorendowmentTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createNameRegistratorPerTxs.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createNameRegistratorPerTxsNotEnoughGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallCreateCallCodeTest/createNameRegistratorPreStore1NotEnoughGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcallcode_001.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcallcode_001_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcallcode_001_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcallcode_001_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcallcode_001_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcallcode_001_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecallcode_011.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecallcode_011_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecallcode_011_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecallcode_011_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecallcode_011_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecallcode_011_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecall_010.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecall_010_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecall_010_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecall_010_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecall_010_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecall_010_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcode_01.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcode_01_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcallcode_01_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcallcode_101.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcallcode_101_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcallcode_101_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcallcode_101_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcallcode_101_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcallcode_101_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcall_100.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcall_100_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcall_100_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcall_100_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcall_100_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcall_100_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecallcode_111.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecallcode_111_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecallcode_111_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecallcode_111_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecallcode_111_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecallcode_111_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecall_110.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecall_110_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecall_110_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecall_110_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecall_110_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecall_110_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcode_11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcode_11_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecallcode_11_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecall_10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecall_10_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead/callcodecall_10_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcallcode_001.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcallcode_001_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcallcode_001_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcallcode_001_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcallcode_001_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcallcode_001_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecallcode_011.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecallcode_011_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecallcode_011_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecallcode_011_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecallcode_011_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecallcode_011_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecall_010.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecall_010_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecall_010_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecall_010_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecall_010_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecall_010_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcode_01.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcode_01_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcallcode_01_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcallcode_101.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcallcode_101_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcallcode_101_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcallcode_101_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcallcode_101_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcallcode_101_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcall_100.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcall_100_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcall_100_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcall_100_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcall_100_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcall_100_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecallcode_111.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecallcode_111_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecallcode_111_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecallcode_111_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecallcode_111_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecallcode_111_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecall_110.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecall_110_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecall_110_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecall_110_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecall_110_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecall_110_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcode_11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcode_11_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecallcode_11_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecall_10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecall_10_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCallDelegateCodesHomestead/callcodecall_10_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/Call1024BalanceTooLow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/Call1024PreCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcallcode_001_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcallcode_001_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcallcode_001_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcall_000_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcodecallcode_011_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcodecallcode_011_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcodecall_010_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcodecall_010_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcallcodecall_010_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcall_00_OOGE_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcall_00_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcall_00_OOGE_valueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/Callcode1024BalanceTooLow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcallcode_101_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcallcode_101_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcallcode_101_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcall_100_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcall_100_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcall_100_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcodecallcode_111_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcodecallcode_111_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcodecallcode_111_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcodecallcode_111_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcodecall_110_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcodecall_110_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/callcodecallcodecall_110_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/contractCreationMakeCallThatAskMoreGasThenTransactionProvided.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stChangedEIP150/createInitFail_OOGduringInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCodeCopyTest/ExtCodeCopyTargetRangeLongerThanCodeTests.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCodeCopyTest/ExtCodeCopyTests.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCodeSizeLimit/codesizeInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCodeSizeLimit/codesizeOOGInvalidSize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCodeSizeLimit/codesizeValid.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/call_outsize_then_create2_successful_then_returndatasize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/call_then_create2_successful_then_returndatasize.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stCreate2/create2callPrecompiles.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2checkFieldsInInitcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionCode2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionNonce.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionSelfdestructed.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionSelfdestructed2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionSelfdestructedOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionSelfdestructedRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2collisionStorage.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2InitCodes.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2noCash.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OnDepth1023.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OnDepth1024.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OOGafterInitCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OOGafterInitCodeReturndata.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OOGafterInitCodeReturndata2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OOGafterInitCodeReturndata3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OOGafterInitCodeReturndataSize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OOGafterInitCodeRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2OOGafterInitCodeRevert2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/Create2Recursive.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/create2SmartInitCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/CREATE2_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/CREATE2_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/CREATE2_Bounds3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/CREATE2_ContractSuicideDuringInit_ThenStoreThenReturn.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/CREATE2_Suicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/CreateMessageReverted.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/CreateMessageRevertedOOGInInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/returndatacopy_0_0_following_successful_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/returndatacopy_afterFailing_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/returndatacopy_following_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/returndatacopy_following_revert_in_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/returndatacopy_following_successful_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/returndatasize_following_successful_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/RevertDepthCreate2OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/RevertDepthCreateAddressCollision.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/RevertInCreateInInitCreate2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/RevertOpcodeCreate.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreate2/RevertOpcodeInCreateReturnsCreate2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateCollisionToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateOOGafterInitCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateOOGafterInitCodeReturndata.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateOOGafterInitCodeReturndata2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateOOGafterInitCodeReturndata3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateOOGafterInitCodeReturndataSize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateOOGafterInitCodeRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CreateOOGafterInitCodeRevert2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_AcreateB_BSuicide_BStore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_ContractRETURNBigOffset.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_ContractSSTOREDuringInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_ContractSuicideDuringInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_ContractSuicideDuringInit_ThenStoreThenReturn.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_ContractSuicideDuringInit_WithValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_ContractSuicideDuringInit_WithValueToItself.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EContractCreateEContractInInit_Tr.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EContractCreateNEContractInInitOOG_Tr.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EContractCreateNEContractInInit_Tr.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EContract_ThenCALLToNonExistentAcc.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_empty000CreateinInitCode_Transaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EmptyContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EmptyContractAndCallIt_0wei.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EmptyContractAndCallIt_1wei.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EmptyContractWithBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EmptyContractWithStorage.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EmptyContractWithStorageAndCallIt_0wei.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/CREATE_EmptyContractWithStorageAndCallIt_1wei.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/TransactionCollisionToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/TransactionCollisionToEmptyButCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stCreateTest/TransactionCollisionToEmptyButNonce.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/Call1024BalanceTooLow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/Call1024OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/Call1024PreCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/CallcodeLoseGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callcodeOutput1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callcodeOutput2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callcodeOutput3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callcodeOutput3Fail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callcodeOutput3partial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callcodeOutput3partialFail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callcodeWithHighValueAndGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/CallLoseGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callOutput1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callOutput2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callOutput3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callOutput3Fail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callOutput3partial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callOutput3partialFail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/CallRecursiveBombPreCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/callWithHighValueAndGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/deleagateCallAfterValueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/Delegatecall1024.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/Delegatecall1024OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallAndOOGatTxLevel.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallBasic.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallEmptycontract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallInInitcodeToEmptyContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallInInitcodeToExistingContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallInInitcodeToExistingContractOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallOOGinCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallSenderCheck.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecallValueCheck.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecodeDynamicCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stDelegatecallTestHomestead/delegatecodeDynamicCode2SelfCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawBalanceGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGasAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGasMemory.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGasMemoryAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGasValueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGasValueTransferAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGasValueTransferMemory.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallCodeGasValueTransferMemoryAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallGasAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallGasValueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallGasValueTransferAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallGasValueTransferMemory.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallGasValueTransferMemoryAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallMemoryGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCallMemoryGasAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCreateFailGasValueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCreateFailGasValueTransfer2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCreateGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCreateGasMemory.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCreateGasValueTransfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawCreateGasValueTransferMemory.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawDelegateCallGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawDelegateCallGasAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawDelegateCallGasMemory.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawDelegateCallGasMemoryAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawExtCodeCopyGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawExtCodeCopyMemoryGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150singleCodeGasPrices/RawExtCodeSizeGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/CallAndCallcodeConsumeMoreGasThenTransactionHas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/CallAskMoreGasOnDepth2ThenTransactionHas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/CallGoesOOGOnSecondLevel.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/CallGoesOOGOnSecondLevel2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/CreateAndGasInsideCreate.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/DelegateCallOnEIP.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/ExecuteCallThatAskForeGasThenTrabsactionHas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/NewGasPriceForCodes.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/SuicideToExistingContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/SuicideToNotExistingContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/Transaction64Rule_d64e0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/Transaction64Rule_d64m1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP150Specific/Transaction64Rule_d64p1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP158Specific/callToEmptyThenCallError.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP158Specific/CALL_OneVCallSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP158Specific/CALL_ZeroVCallSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP158Specific/EXP_Empty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP158Specific/EXTCODESIZE_toEpmty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP158Specific/EXTCODESIZE_toNonExistent.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stEIP158Specific/vitalikTransactionTest.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExample/add11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashAccountWithoutCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashCALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashCALLCODE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashDELEGATECALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashDynamicArgument.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashInInitCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashMaxCodeSize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashNewAccount.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashNonExistingAccount.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashPrecompiles.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashSelf.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashSelfInInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stExtCodeHash/extCodeHashSTATICCALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stHomesteadSpecific/contractCreationOOGdontLeaveEmptyContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stHomesteadSpecific/contractCreationOOGdontLeaveEmptyContractViaTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stHomesteadSpecific/createContractViaContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stHomesteadSpecific/createContractViaContractOOGInitCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stHomesteadSpecific/createContractViaTransactionCost53000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallContractToCreateContractAndCallItOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallContractToCreateContractNoCash.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallContractToCreateContractOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallContractToCreateContractOOGBonusGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallContractToCreateContractWhichWouldCreateContractIfCalled.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallContractToCreateContractWhichWouldCreateContractInInitCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallRecursiveContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/CallTheContractToCreateEmptyContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/NotEnoughCashContractCreation.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/OutOfGasContractCreation.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/OutOfGasPrefundedContractCreation.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/ReturnTest.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/ReturnTest2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/StackUnderFlowContractCreation.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/TransactionCreateAutoSuicideContract.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/TransactionCreateRandomInitCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/TransactionCreateStopInInitcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stInitCodeTest/TransactionCreateSuicideInInitcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log0_emptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log0_logMemsizeTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log0_logMemsizeZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log0_logMemStartTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log0_nonEmptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log0_nonEmptyMem_logMemSize1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log0_nonEmptyMem_logMemSize1_logMemStart31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_Caller.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_emptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_logMemsizeTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_logMemsizeZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_logMemStartTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_MaxTopic.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_nonEmptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_nonEmptyMem_logMemSize1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log1_nonEmptyMem_logMemSize1_logMemStart31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_Caller.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_emptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_logMemsizeTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_logMemsizeZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_logMemStartTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_MaxTopic.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_nonEmptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_nonEmptyMem_logMemSize1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log2_nonEmptyMem_logMemSize1_logMemStart31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_Caller.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_emptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_logMemsizeTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_logMemsizeZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_logMemStartTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_MaxTopic.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_nonEmptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_nonEmptyMem_logMemSize1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_nonEmptyMem_logMemSize1_logMemStart31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log3_PC.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_Caller.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_emptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_logMemsizeTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_logMemsizeZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_logMemStartTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_MaxTopic.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_nonEmptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_nonEmptyMem_logMemSize1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_nonEmptyMem_logMemSize1_logMemStart31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/log4_PC.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stLogTests/logInOOG_Call.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/CallAndCallcodeConsumeMoreGasThenTransactionHasWithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/CallAskMoreGasOnDepth2ThenTransactionHasWithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/CallGoesOOGOnSecondLevel2WithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/CallGoesOOGOnSecondLevelWithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/CreateAndGasInsideCreateWithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/DelegateCallOnEIPWithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/ExecuteCallThatAskMoreGasThenTransactionHasWithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemExpandingEIP150Calls/NewGasPriceForCodesWithMemExpandingCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALLCODE_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALLCODE_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALLCODE_Bounds3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALLCODE_Bounds4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALL_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALL_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALL_Bounds2a.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CALL_Bounds3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CREATE_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CREATE_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/CREATE_Bounds3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/DELEGATECALL_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/DELEGATECALL_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/DELEGATECALL_Bounds3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/DUP_Bounds.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryStressTest/FillStack.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryStressTest/JUMPI_Bounds.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryStressTest/JUMP_Bounds.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryStressTest/JUMP_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/mload32bitBound.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/mload32bitBound2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/mload32bitBound_Msize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/mload32bitBound_return.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/mload32bitBound_return2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/MLOAD_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/MLOAD_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/MLOAD_Bounds3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/MSTORE_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/MSTORE_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/MSTORE_Bounds2a.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/POP_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/RETURN_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/SLOAD_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/SSTORE_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/static_CALL_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/static_CALL_Bounds2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/static_CALL_Bounds2a.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryStressTest/static_CALL_Bounds3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/callDataCopyOffset.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/calldatacopy_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/calldatacopy_dejavu2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/codeCopyOffset.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/codecopy_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/codecopy_dejavu2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/extcodecopy_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/log1_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/log2_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/log3_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/log4_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem0b_singleByte.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem31b_singleByte.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32b_singleByte.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb+1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb+31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb+32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb+33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb-1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb-31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb-32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb-33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte+1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte+31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte+32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte+33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte-1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte-31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte-32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte-33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem32kb_singleByte.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem33b_singleByte.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb+1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb+31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb+32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb+33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb-1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb-31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb-32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb-33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte+1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte+31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte+32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte+33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte-1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte-31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte-32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte-33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mem64kb_singleByte.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/memReturn.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mload16bitBound.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mload8bitBound.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mload_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mstore_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/mstroe8_dejavu.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stMemoryTest/sha3_dejavu.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitGas_1023.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitGas_1024.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitGas_1025.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitPush31_1023.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitPush31_1024.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitPush31_1025.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitPush32_1023.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitPush32_1024.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stMemoryTest/stackLimitPush32_1025.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALLCODE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALLCODE_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALLCODE_ToNonNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALLCODE_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALL_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALL_ToNonNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_CALL_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_DELEGATECALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_DELEGATECALL_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_DELEGATECALL_ToNonNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_DELEGATECALL_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_SUICIDE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_SUICIDE_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_SUICIDE_ToNonNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_SUICIDE_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALLwithData.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALLwithData_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALLwithData_ToNonNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALLwithData_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALL_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALL_ToNonNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stNonZeroCallsTest/NonZeroValue_TransactionCALL_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/identity_to_bigger.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/identity_to_smaller.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_0_0_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_0_0_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_1_0_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_1_0_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_1_0_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_1_0_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_1_0_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_1_0_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_3_100_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_3_100_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_3_100_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_3_100_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_3_100_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_0_3_100_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_0_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_0_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_0_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_0_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_0_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_0_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_1_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_1_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_1_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_1_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_1_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_0_1_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_1_1_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_1_1_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_1_1_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_1_1_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_1_1_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_1_1_1_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_22411_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_0_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_0_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_0_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_0_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_0_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_0_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_1_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_1_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_1_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_1_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_1_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_37111_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_37111_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_37111_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_37111_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_37111_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_37111_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_97_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_97_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_97_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_97_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_97_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_37120_37111_97_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_39936_1_55201_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_39936_1_55201_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_39936_1_55201_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_39936_1_55201_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_39936_1_55201_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_39936_1_55201_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_09984_39936_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_09984_39936_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_09984_39936_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_09984_39936_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_09984_39936_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_28948_11579_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_5_100_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_5_100_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_5_100_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_5_100_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_5_100_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_3_5_100_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_49_2401_2401_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_49_2401_2401_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_49_2401_2401_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_49_2401_2401_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_49_2401_2401_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_49_2401_2401_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_55190_55190_42965_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_55190_55190_42965_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_55190_55190_42965_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_55190_55190_42965_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_55190_55190_42965_25000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_55190_55190_42965_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_9_37111_37111_1000000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_9_37111_37111_155000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_9_37111_37111_20500.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_9_37111_37111_22000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_9_37111_37111_35000.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/modexp_9_3711_37111_25000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts/sec80.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover0_0input.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover0_completeReturnValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover0_Gas2999.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover0_gas3000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover0_NoGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover0_overlappingInputOutput.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecover80.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecoverH_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecoverR_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecoverS_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecoverV_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEEcrecoverV_prefixedf0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentitiy_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentitiy_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentity_1_nonzeroValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentity_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentity_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentity_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentity_4_gas17.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentity_4_gas18.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODEIdentity_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_3_postfixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_3_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_4_gas719.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODERipemd160_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_1_nonzeroValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_3_postfix0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_3_prefix0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_4_gas99.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CALLCODESha256_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover0_0input.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover0_completeReturnValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover0_Gas2999.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover0_gas3000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover0_NoGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover0_overlappingInputOutput.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecover80.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverCheckLength.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverCheckLengthWrongV.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverH_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverInvalidSignature.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverR_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverS_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverUnrecoverableKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallEcrecoverV_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentitiy_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentitiy_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_1_nonzeroValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_4_gas17.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_4_gas18.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallIdentity_6_inputShorterThanOutput.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_3_postfixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_3_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_4_gas719.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallRipemd160_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_1_nonzeroValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_3_postfix0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_3_prefix0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_4_gas99.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/CallSha256_5.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/modexpRandomInput.json");
        skip("precompile","/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/modexp_0_0_0_20500.json");
        skip("precompile","/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/modexp_0_0_0_22000.json");
        skip("precompile","/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/modexp_0_0_0_25000.json");
        skip("precompile","/tmp/jsondata/GeneralStateTests/stPreCompiledContracts2/modexp_0_0_0_35000.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call1MB1024Calldepth.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000bytesContract50_1.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000bytesContract50_2.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000bytesContract50_3.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000_ecrec.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000_identity.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000_identity2.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000_rip160.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Call50000_sha256.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Callcode50000.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Create1000.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Create1000Byzantium.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/QuadraticComplexitySolidity_CallDataCopy.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Return50000.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stQuadraticComplexityTest/Return50000_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest100.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest101.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest102.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest103.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest104.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest105.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest106.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest107.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest108.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest110.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest111.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest112.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest114.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest115.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest116.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest117.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest118.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest119.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest12.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest120.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest121.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest122.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest123.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest124.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest125.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest126.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest129.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest13.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest130.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest131.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest133.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest134.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest135.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest136.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest137.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest138.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest139.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest14.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest142.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest143.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest144.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest145.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest146.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest147.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest148.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest149.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest15.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest150.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest151.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest153.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest154.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest155.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest156.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest157.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest158.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest159.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest16.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest160.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest161.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest162.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest163.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest164.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest166.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest167.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest169.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest17.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest170.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest171.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest172.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest173.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest174.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest175.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest176.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest177.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest178.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest179.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest18.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest180.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest183.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest184.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest185.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest187.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest188.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest189.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest19.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest190.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest191.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest192.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest194.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest195.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest196.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest197.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest198.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest199.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest20.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest200.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest201.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest202.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest204.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest205.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest206.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest207.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest208.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest209.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest210.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest211.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest212.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest214.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest215.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest216.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest217.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest219.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest22.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest220.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest221.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest222.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest223.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest225.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest226.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest227.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest228.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest229.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest23.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest230.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest231.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest232.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest233.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest236.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest237.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest238.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest24.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest241.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest242.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest243.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest244.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest245.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest246.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest247.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest248.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest249.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest25.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest250.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest251.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest252.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest254.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest257.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest259.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest26.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest260.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest261.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest263.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest264.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest265.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest266.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest267.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest268.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest269.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest27.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest270.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest271.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest273.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest274.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest275.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest276.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest278.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest279.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest28.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest280.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest281.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest282.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest283.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest285.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest286.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest287.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest288.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest29.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest290.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest291.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest292.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest293.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest294.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest295.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest296.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest297.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest298.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest299.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest30.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest300.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest301.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest302.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest303.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest304.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest305.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest306.json");
        skip("obsolete", "/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest307.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest308.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest309.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest310.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest311.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest312.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest313.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest315.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest316.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest318.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest32.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest320.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest321.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest322.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest323.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest324.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest325.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest326.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest327.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest328.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest329.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest33.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest332.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest333.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest334.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest335.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest336.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest337.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest338.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest339.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest340.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest341.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest342.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest343.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest345.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest346.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest347.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest348.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest349.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest350.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest351.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest352.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest353.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest354.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest355.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest356.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest357.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest358.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest359.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest36.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest360.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest361.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest362.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest363.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest364.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest365.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest366.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest367.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest368.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest369.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest37.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest370.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest371.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest372.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest375.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest376.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest377.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest378.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest379.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest38.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest380.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest381.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest382.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest383.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest39.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest41.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest42.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest43.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest45.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest46.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest47.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest48.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest49.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest50.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest51.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest52.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest53.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest54.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest55.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest57.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest58.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest59.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest6.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest60.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest62.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest63.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest64.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest66.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest67.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest69.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest7.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest72.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest73.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest74.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest75.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest77.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest78.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest80.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest81.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest82.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest83.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest84.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest85.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest87.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest88.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest89.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest9.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest90.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest92.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest94.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest95.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest96.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest97.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom/randomStatetest98.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/201503110226PYTHON_DUP6.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest384.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest385.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest386.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest387.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest388.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest389.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest391.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest393.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest395.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest396.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest397.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest398.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest399.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest401.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest402.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest404.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest405.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest406.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest407.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest408.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest409.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest410.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest411.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest412.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest413.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest414.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest415.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest416.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest417.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest418.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest419.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest420.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest421.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest422.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest423.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest424.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest425.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest426.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest428.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest429.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest430.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest433.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest435.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest436.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest437.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest438.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest439.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest440.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest441.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest442.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest443.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest444.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest445.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest446.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest447.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest448.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest449.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest450.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest451.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest452.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest454.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest455.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest456.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest457.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest458.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest460.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest461.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest462.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest464.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest465.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest466.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest467.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest468.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest469.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest470.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest471.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest472.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest473.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest474.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest475.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest476.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest477.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest478.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest480.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest481.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest482.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest483.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest484.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest485.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest487.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest488.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest489.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest491.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest493.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest494.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest495.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest496.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest497.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest498.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest499.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest500.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest501.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest502.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest503.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest504.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest505.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest506.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest507.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest508.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest509.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest510.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest511.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest512.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest513.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest514.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest516.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest517.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest518.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest519.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest520.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest521.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest523.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest524.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest525.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest526.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest527.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest528.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest531.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest532.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest533.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest534.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest535.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest536.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest537.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest538.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest539.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest541.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest542.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest543.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest544.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest545.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest546.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest547.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest548.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest549.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest550.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest552.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest553.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest554.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest555.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest556.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest558.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest559.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest560.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest562.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest563.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest564.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest565.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest566.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest567.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest569.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest571.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest572.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest573.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest574.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest575.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest576.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest577.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest578.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest579.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest580.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest581.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest582.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest583.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest584.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest585.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest586.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest587.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest588.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest589.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest592.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest594.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest596.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest597.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest599.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest600.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest601.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest602.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest603.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest604.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest605.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest607.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest608.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest609.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest610.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest611.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest612.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest615.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest616.json");
        skip("obsolete", "/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest618.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest619.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest620.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest621.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest624.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest625.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest626.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest627.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest628.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest629.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest630.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest632.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest633.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest635.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest636.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest637.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest638.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest639.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest640.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest641.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest642.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest643.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest644.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest645.json");
        skip("obsolete", "/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest646.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRandom2/randomStatetest647.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRecursiveCreate/recursiveCreate.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRecursiveCreate/recursiveCreateReturnValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund50percentCap.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund50_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund50_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund600.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refundSuicide50procentCap.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_CallA.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_CallA_notEnoughGasInCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_CallA_OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_CallToSuicideNoStorage.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_CallToSuicideStorage.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_CallToSuicideTwice.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_changeNonZeroStorage.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_getEtherBack.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_multimpleSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_NoOOG_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_singleSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_TxToSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRefundTest/refund_TxToSuicideOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/call_ecrec_success_empty_then_returndatasize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/call_outsize_then_create_successful_then_returndatasize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/call_then_call_value_fail_then_returndatasize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/call_then_create_successful_then_returndatasize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/create_callprecompile_returndatasize.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stReturnDataTest/modexp_modsize0_returndatasize.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_0_0_following_successful_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_afterFailing_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_after_failing_callcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_after_failing_delegatecall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_after_failing_staticcall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_after_revert_in_staticcall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_after_successful_callcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_after_successful_delegatecall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_after_successful_staticcall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_following_call.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_following_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_following_failing_call.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_following_revert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_following_revert_in_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_following_successful_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_following_too_big_transfer.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_initial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_initial_256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_initial_big_sum.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatacopy_overrun.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_after_failing_callcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_after_failing_delegatecall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_after_failing_staticcall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_after_oog_after_deeper.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_after_successful_callcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_after_successful_delegatecall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_after_successful_staticcall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_bug.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_following_successful_create.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_initial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stReturnDataTest/returndatasize_initial_zero_read.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/LoopCallsDepthThenRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/LoopCallsDepthThenRevert2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/LoopCallsDepthThenRevert3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/LoopCallsThenRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/LoopDelegateCallsDepthThenRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/NashatyrevSuicideRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/PythonRevertTestTue201814-1430.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertDepth2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertDepthCreateAddressCollision.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertDepthCreateOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertInCallCode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertInCreateInInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertInDelegateCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertInStaticCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOnEmptyStack.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeCreate.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeDirectCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeInCallsOnNonEmptyReturnData.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeInCreateReturns.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeInInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeMultipleSubCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeReturn.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertOpcodeWithBigOutputInInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrecompiledTouch.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrecompiledTouchCC.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrecompiledTouchDC.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefound.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefoundCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefoundCallOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefoundEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefoundEmptyCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefoundEmptyCallOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefoundEmptyOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertPrefoundOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertRemoteSubCallStorageOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertRemoteSubCallStorageOOG2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertSubCallStorageOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/RevertSubCallStorageOOG2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/TouchToEmptyAccountRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/TouchToEmptyAccountRevert2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stRevertTest/TouchToEmptyAccountRevert3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar00.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar01.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_0_256-1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^254_254.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255-1_248.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255-1_254.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255-1_255.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255-1_256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255_255.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255_256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^255_257.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^256-1_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^256-1_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^256-1_255.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/sar_2^256-1_256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl01-0100.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl01-0101.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl01-ff.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl01.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl_-1_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl_-1_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl_-1_255.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl_-1_256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shl_2^255-1_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr01.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr11.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_-1_0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_-1_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_-1_255.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_-1_256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_2^255_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_2^255_255.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_2^255_256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stShift/shr_2^255_257.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/AmbiguousMethod.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/CallInfiniteLoop.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/CallLowLevelCreatesSolidity.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/CallRecursiveMethods.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/ContractInheritance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/CreateContractFromMethod.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/RecursiveCreateContracts.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/RecursiveCreateContractsCreate4Contracts.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestBlockAndTransactionProperties.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestContractInteraction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestContractSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestCryptographicFunctions.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestKeywords.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestOverflow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestStoreGasPrices.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSolidityTest/TestStructuresAndVariabless.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/block504980.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/deploymentError.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/FailedCreateRevertsDeletion.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/failed_tx_xcf416c53.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/gasPrice0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/JUMPDEST_Attack.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/JUMPDEST_AttackwithJump.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/makeMoney.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/OverflowGasMakeMoney.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/push32withoutByte.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/sha3_deja.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/StackDepthLimitSEC.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/txCost-sec73.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSpecialTest/tx_e1c174e2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/InitCollision.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/SstoreCallToSelfSubRefundBelowZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0to0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0to0to0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0to0toX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0toX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0toXto0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0toXto0toX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0toXtoX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_0toXtoY.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_changeFromExternalCallInInitCode.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_combinations_initial0.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_combinations_initial1.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_combinations_initial2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_Xto0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_Xto0to0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_Xto0toX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_Xto0toXto0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_Xto0toY.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoXto0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoXtoX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoXtoY.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoY.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoYto0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoYtoX.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoYtoY.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSStoreTest/sstore_XtoYtoZ.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStackTests/shallowStack.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStackTests/stackOverflow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStackTests/stackOverflowDUP.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStackTests/stackOverflowM1.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stStackTests/stackOverflowM1DUP.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stStackTests/stackOverflowM1PUSH.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStackTests/stackOverflowPUSH.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ABAcalls0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ABAcalls1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ABAcalls2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ABAcalls3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ABAcallsSuicide0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ABAcallsSuicide1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call1024BalanceTooLow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call1024BalanceTooLow2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call1024OOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call1024PreCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call1024PreCalls2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call1024PreCalls3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call1MB1024Calldepth.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000bytesContract50_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000bytesContract50_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000bytesContract50_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000_ecrec.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000_identity.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000_identity2.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000_rip160.json");
        skip("slow", "/tmp/jsondata/GeneralStateTests/stStaticCall/static_Call50000_sha256.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallAndCallcodeConsumeMoreGasThenTransactionHas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallAskMoreGasOnDepth2ThenTransactionHas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callBasic.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGMAfter2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_OOGMBefore2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_001_SuicideMiddle2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcallcode_ABCB_RECURSIVE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_000_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_000_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_000_OOGMAfter2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_000_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_000_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_000_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGMAfter2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_OOGMBefore2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_011_SuicideMiddle2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecallcode_ABCB_RECURSIVE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGMAfter2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_OOGMBefore2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_010_SuicideMiddle2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcodecall_ABCB_RECURSIVE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcode_01_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcode_01_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcode_01_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcallcode_01_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcall_00.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcall_00_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcall_00_OOGE_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcall_00_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcall_00_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callChangeRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGMAfter2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGMAfter_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_OOGMBefore2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_101_SuicideMiddle2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcallcode_ABCB_RECURSIVE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGMAfter2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_OOGMBefore2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_100_SuicideMiddle2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcall_ABCB_RECURSIVE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecallcode_111_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_1102.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGMAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGMAfter2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGMAfter_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGMAfter_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGMBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_OOGMBefore2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_SuicideMiddle.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_110_SuicideMiddle2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_ABCB_RECURSIVE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecallcodecall_ABCB_RECURSIVE2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecall_10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecall_10_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecall_10_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecall_10_OOGE_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecall_10_SuicideEnd.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcodecall_10_SuicideEnd2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callcode_checkPC.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallContractToCreateContractAndCallItOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallContractToCreateContractOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallContractToCreateContractOOGBonusGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallContractToCreateContractWhichWouldCreateContractIfCalled.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callCreate.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callCreate2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callCreate3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_calldelcode_01.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_calldelcode_01_OOGE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover0.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover0_0input.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover0_completeReturnValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover0_Gas2999.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover0_gas3000.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover0_NoGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover0_overlappingInputOutput.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecover80.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecoverCheckLength.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecoverCheckLengthWrongV.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecoverH_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecoverR_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecoverS_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallEcrecoverV_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallGoesOOGOnSecondLevel.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallGoesOOGOnSecondLevel2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentitiy_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentity_1_nonzeroValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentity_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentity_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentity_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentity_4_gas17.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentity_4_gas18.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallIdentity_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallLoseGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callOutput1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callOutput2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callOutput3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callOutput3Fail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callOutput3partial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callOutput3partialFail.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBomb0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBomb0_OOG_atMaxCallDepth.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBomb1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBomb2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBomb3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBombLog.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBombLog2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBombPreCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRecursiveBombPreCall2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_3_postfixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_3_prefixed0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_4_gas719.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallRipemd160_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_1_nonzeroValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_3_postfix0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_3_prefix0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_4_gas99.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallSha256_5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callToCallCodeOpCodeCheck.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callToCallOpCodeCheck.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callToDelCallOpCodeCheck.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallToNameRegistrator0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CallToReturn1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CalltoReturn2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callToStaticOpCodeCheck.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callWithHighValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callWithHighValueAndGasOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callWithHighValueAndOOGatTxLevel.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_callWithHighValueOOGinCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CALL_OneVCallSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_call_OOG_additionalGasCosts1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_call_OOG_additionalGasCosts2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_call_value_inherit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_call_value_inherit_from_call.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CALL_ZeroVCallSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CheckCallCostOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CheckOpcodes.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CheckOpcodes2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CheckOpcodes3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CheckOpcodes4.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CheckOpcodes5.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_contractCreationMakeCallThatAskMoreGasThenTransactionProvided.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_contractCreationOOGdontLeaveEmptyContractViaTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CREATE_ContractSuicideDuringInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CREATE_ContractSuicideDuringInit_ThenStoreThenReturn.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CREATE_ContractSuicideDuringInit_WithValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CREATE_EmptyContractAndCallIt_0wei.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_CREATE_EmptyContractWithStorageAndCallIt_0wei.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ExecuteCallThatAskForeGasThenTrabsactionHas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_InternalCallHittingGasLimit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_InternalCallHittingGasLimit2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_InternlCallStoreClearsOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log0_emptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log0_logMemsizeTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log0_logMemsizeZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log0_logMemStartTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log0_nonEmptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log0_nonEmptyMem_logMemSize1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log0_nonEmptyMem_logMemSize1_logMemStart31.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log1_emptyMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log1_logMemsizeTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log1_logMemsizeZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log1_logMemStartTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log1_MaxTopic.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_log_Caller.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_LoopCallsDepthThenRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_LoopCallsDepthThenRevert2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_LoopCallsDepthThenRevert3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_LoopCallsThenRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_makeMoney.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_PostToReturn1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_RawCallGasAsk.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_refund_CallA.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_refund_CallToSuicideNoStorage.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_refund_CallToSuicideTwice.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_Return50000_2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ReturnTest.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ReturnTest2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_RETURN_Bounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_RETURN_BoundsOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_RevertDepth2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_RevertOpcodeCalls.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ZeroValue_CALL_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stStaticCall/static_ZeroValue_SUICIDE_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/ABAcalls0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/ABAcalls1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/ABAcalls2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/ABAcalls3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/ABAcallsSuicide0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/ABAcallsSuicide1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/balanceInputAddressTooBig.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/Call10.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callcodeTo0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callcodeToNameRegistrator0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callcodeToNameRegistratorAddresTooBigLeft.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callcodeToNameRegistratorAddresTooBigRight.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callcodeToNameRegistratorZeroMemExpanion.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callcodeToReturn1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callerAccountBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallRecursiveBomb0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallRecursiveBomb0_OOG_atMaxCallDepth.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallRecursiveBomb1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallRecursiveBomb2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallRecursiveBomb3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallRecursiveBombLog.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallRecursiveBombLog2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistrator0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorAddressTooBigLeft.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorAddressTooBigRight.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorMemOOGAndInsufficientBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorNotMuchMemory0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorNotMuchMemory1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorOutOfGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorTooMuchMemory0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorTooMuchMemory1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorTooMuchMemory2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToNameRegistratorZeorSizeMemExpansion.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToReturn1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToReturn1ForDynamicJump0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CallToReturn1ForDynamicJump1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CalltoReturn2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/callValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/CreateHashCollision.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistrator.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistratorOOG_MemExpansionOOV.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistratorOutOfMemoryBonds0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistratorOutOfMemoryBonds1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistratorValueTooHigh.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistratorZeroMem.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistratorZeroMem2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createNameRegistratorZeroMemExpansion.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/createWithInvalidOpcode.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/currentAccountBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/doubleSelfdestructTest.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/doubleSelfdestructTest2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/extcodecopy.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/PostToReturn1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/return0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/return1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/return2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideAddress.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideCaller.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideCallerAddresTooBigLeft.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideCallerAddresTooBigRight.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideCoinbase.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideNotExistingAccount.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideOrigin.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideSendEtherPostDeath.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/suicideSendEtherToMe.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/TestNameRegistrator.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stSystemOperationsTest/testRandomTest.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/ContractStoreClearsOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/ContractStoreClearsSuccess.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/CreateMessageReverted.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/CreateMessageSuccess.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/CreateTransactionReverted.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/CreateTransactionSuccess.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/EmptyTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/EmptyTransaction2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/EmptyTransaction3.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/HighGasLimit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/InternalCallHittingGasLimit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/InternalCallHittingGasLimit2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/InternalCallHittingGasLimitSuccess.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/InternlCallStoreClearsOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/InternlCallStoreClearsSucces.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/Opcodes_TransactionInit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/OverflowGasRequire.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/OverflowGasRequire2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/RefundOverflow.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/RefundOverflow2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/StoreClearsAndInternlCallStoreClearsOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/StoreClearsAndInternlCallStoreClearsSuccess.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/StoreGasOnCreate.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/SuicidesAndInternlCallSuicidesBonusGasAtCall.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/SuicidesAndInternlCallSuicidesBonusGasAtCallFailed.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/SuicidesAndInternlCallSuicidesOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/SuicidesAndInternlCallSuicidesSuccess.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/SuicidesAndSendMoneyToItselfEtherDestroyed.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/SuicidesMixingCoinbase.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/SuicidesStopAfterSuicide.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionDataCosts652.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionFromCoinbaseHittingBlockGasLimit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionFromCoinbaseHittingBlockGasLimit1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionFromCoinbaseNotEnoughFounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionNonceCheck.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionNonceCheck2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionSendingToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionSendingToZero.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionToAddressh160minusOne.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionToItself.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/TransactionToItselfNotEnoughFounds.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/UserTransactionGasLimitIsTooLowWhenZeroCost.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/UserTransactionZeroCost.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransactionTest/UserTransactionZeroCostWithData.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/createNameRegistratorPerTxsAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/createNameRegistratorPerTxsAt.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/createNameRegistratorPerTxsBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/createNameRegistratorPerTxsNotEnoughGasAfter.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/createNameRegistratorPerTxsNotEnoughGasAt.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/createNameRegistratorPerTxsNotEnoughGasBefore.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/delegatecallAfterTransition.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/delegatecallAtTransition.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stTransitionTest/delegatecallBeforeTransition.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/dayLimitConstruction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/dayLimitConstructionOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/dayLimitConstructionPartial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/dayLimitResetSpentToday.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/dayLimitSetDailyLimit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/dayLimitSetDailyLimitNoData.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedAddOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedAddOwnerAddMyself.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedChangeOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedChangeOwnerNoArgument.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedChangeOwner_fromNotOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedChangeOwner_toIsOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedChangeRequirementTo0.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedChangeRequirementTo1.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedChangeRequirementTo2.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedConstructionCorrect.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedConstructionNotEnoughGas.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedConstructionNotEnoughGasPartial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedIsOwnerFalse.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedIsOwnerTrue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedRemoveOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedRemoveOwnerByNonOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedRemoveOwner_mySelf.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedRemoveOwner_ownerIsNotOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/multiOwnedRevokeNothing.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletAddOwnerRemovePendingTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletChangeOwnerRemovePendingTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletChangeRequirementRemovePendingTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletConfirm.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletConstruction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletConstructionOOG.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletConstructionPartial.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletDefault.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletDefaultWithOutValue.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletExecuteOverDailyLimitMultiOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletExecuteOverDailyLimitOnlyOneOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletExecuteOverDailyLimitOnlyOneOwnerNew.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletExecuteUnderDailyLimit.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletKill.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletKillNotByOwner.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletKillToWallet.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stWalletTest/walletRemoveOwnerRemovePendingTransaction.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALLCODE_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALLCODE_ToEmpty_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALLCODE_ToNonZeroBalance_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALLCODE_ToOneStorageKey_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALL_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALL_ToEmpty_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALL_ToNonZeroBalance_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_CALL_ToOneStorageKey_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_DELEGATECALL_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_DELEGATECALL_ToEmpty_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_DELEGATECALL_ToNonZeroBalance_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_DELEGATECALL_ToOneStorageKey_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_SUICIDE_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_SUICIDE_ToEmpty_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_SUICIDE_ToNonZeroBalance_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_SUICIDE_ToOneStorageKey_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALLwithData_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALLwithData_ToEmpty_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALLwithData_ToNonZeroBalance_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALLwithData_ToOneStorageKey_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALL_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALL_ToEmpty_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALL_ToNonZeroBalance_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsRevert/ZeroValue_TransactionCALL_ToOneStorageKey_OOGRevert.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALLCODE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALLCODE_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALLCODE_ToNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALLCODE_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALL_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALL_ToNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_CALL_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_DELEGATECALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_DELEGATECALL_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_DELEGATECALL_ToNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_DELEGATECALL_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_SUICIDE.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_SUICIDE_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_SUICIDE_ToNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_SUICIDE_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALL.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALLwithData.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALLwithData_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALLwithData_ToNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALLwithData_ToOneStorageKey.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALL_ToEmpty.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALL_ToNonZeroBalance.json");
        test_json_file("/tmp/jsondata/GeneralStateTests/stZeroCallsTest/ZeroValue_TransactionCALL_ToOneStorageKey.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_2_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_2_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_340282366920938463463374607431768211456_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_340282366920938463463374607431768211456_21000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_340282366920938463463374607431768211456_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_340282366920938463463374607431768211456_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_340282366920938463463374607431768211456_28000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_340282366920938463463374607431768211456_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_5616_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_5616_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_5616_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_5617_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_5617_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_5617_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_5617_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_616_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9935_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9935_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9935_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9935_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-2_9_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_21000_64.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_21000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_28000_64.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_28000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_0_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_1_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_1_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_1_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_1_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_2_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_2_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_2_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_2_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_340282366920938463463374607431768211456_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_340282366920938463463374607431768211456_21000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_340282366920938463463374607431768211456_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_340282366920938463463374607431768211456_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_340282366920938463463374607431768211456_28000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_340282366920938463463374607431768211456_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5616_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5616_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5616_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5616_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5617_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5617_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5617_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_5617_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9935_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9935_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9935_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9935_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_1-3_9_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_21000_64.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_21000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_28000_64.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_28000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_0_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1456_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1456_21000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1456_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1456_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1456_28000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1456_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_1_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_2_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_2_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_2_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_2_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5616_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5616_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5616_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5616_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5617_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5617_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5617_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_5617_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9935_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9935_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9935_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9935_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9_21000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9_28000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecmul_7827-6598_9_28000_96.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_bad_length_191.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_bad_length_193.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_empty_data.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_empty_data_insufficient_gas.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_one_point_fail.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_one_point_insufficient_gas.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_one_point_not_in_subgroup.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_one_point_with_g1_zero.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_one_point_with_g2_zero.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_one_point_with_g2_zero_and_g1_invalid.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_perturb_g2_by_curve_order.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_perturb_g2_by_field_modulus.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_perturb_g2_by_field_modulus_again.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_perturb_g2_by_one.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_perturb_zeropoint_by_curve_order.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_perturb_zeropoint_by_field_modulus.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_perturb_zeropoint_by_one.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_three_point_fail_1.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_three_point_match_1.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_points_with_one_g2_zero.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_fail_1.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_fail_2.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_match_1.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_match_2.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_match_3.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_match_4.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_match_5.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/ecpairing_two_point_oog.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/pairingTest.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/pointAdd.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/pointAddTrunc.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/pointMulAdd.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge/pointMulAdd2.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_21000_0.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_21000_192.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_21000_64.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_21000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_25000_0.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_25000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_25000_192.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_25000_64.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_0-0_25000_80.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_1-2_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_1-2_21000_192.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_1-2_25000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_1-2_25000_192.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_1-3_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-0_1-3_25000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-3_1-2_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_0-3_1-2_25000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_1-2_0-0_21000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_1-2_0-0_21000_192.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_1-2_0-0_21000_64.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_1-2_0-0_25000_128.json");
        skip("precompile", "/tmp/jsondata/GeneralStateTests/stZeroKnowledge2/ecadd_1-2_0-0_25000_192.json");*/

        test_json_path(r"../jsondata/GeneralStateTests/stRandom");
        test_json_path(r"../jsondata/GeneralStateTests/stSystemOperationsTest");
        test_json_path(r"../jsondata/GeneralStateTests/stRecursiveCreate");
        test_json_path(r"../jsondata/GeneralStateTests/stLogTests");
        test_json_path(r"../jsondata/GeneralStateTests/stCodeCopyTest");
        test_json_path(r"../jsondata/GeneralStateTests/stExtCodeHash");
        test_json_path(r"../jsondata/GeneralStateTests/stCallCodes");
        test_json_path(r"../jsondata/GeneralStateTests/stCreateTest");
        test_json_path(r"../jsondata/GeneralStateTests/stZeroKnowledge");
        test_json_path(r"../jsondata/GeneralStateTests/stRandom2");
        test_json_path(r"../jsondata/GeneralStateTests/stTransitionTest");
        test_json_path(r"../jsondata/GeneralStateTests/stZeroCallsTest");
        test_json_path(r"../jsondata/GeneralStateTests/stBugs");
        test_json_path(r"../jsondata/GeneralStateTests/stBadOpcode");
        test_json_path(r"../jsondata/GeneralStateTests/stWalletTest");
        test_json_path(r"../jsondata/GeneralStateTests/stNonZeroCallsTest");
        test_json_path(r"../jsondata/GeneralStateTests/stCallDelegateCodesHomestead");
        test_json_path(r"../jsondata/GeneralStateTests/stAttackTest");
        test_json_path(r"../jsondata/GeneralStateTests/stStackTests");
        test_json_path(r"../jsondata/GeneralStateTests/stExample");
        test_json_path(r"../jsondata/GeneralStateTests/stSolidityTest");
        test_json_path(r"../jsondata/GeneralStateTests/stQuadraticComplexityTest");
        test_json_path(r"../jsondata/GeneralStateTests/stPreCompiledContracts2");
        test_json_path(r"../jsondata/GeneralStateTests/stInitCodeTest");
        test_json_path(r"../jsondata/GeneralStateTests/stDelegatecallTestHomestead");
        test_json_path(r"../jsondata/GeneralStateTests/stMemoryTest");
        test_json_path(r"../jsondata/GeneralStateTests/stSpecialTest");
        test_json_path(r"../jsondata/GeneralStateTests/stShift");
        test_json_path(r"../jsondata/GeneralStateTests/stHomesteadSpecific");
        test_json_path(r"../jsondata/GeneralStateTests/stCodeSizeLimit");
        test_json_path(r"../jsondata/GeneralStateTests/stReturnDataTest");
        test_json_path(r"../jsondata/GeneralStateTests/stTransactionTest");
        test_json_path(r"../jsondata/GeneralStateTests/stRevertTest");
        test_json_path(r"../jsondata/GeneralStateTests/stMemoryStressTest");
        test_json_path(r"../jsondata/GeneralStateTests/stCallDelegateCodesCallCodeHomestead");
        test_json_path(r"../jsondata/GeneralStateTests/stMemExpandingEIP150Calls");
        test_json_path(r"../jsondata/GeneralStateTests/stRefundTest");
    }).unwrap().join().unwrap();
}
