use crate::models::{ColumnIndex, CsvType, Query, QueryFlags, Table};
use crate::{consts, CIndexError, CIndexResult};
#[cfg(feature = "rayon")]
use rayon::prelude::*;
use std::collections::HashMap;
use std::io::Write;
use std::{fs::File, io::Read};

/// Entry struct for indexing csv tables
pub struct Indexer {
    tables: HashMap<String, Table>,
    use_unix_newline: bool,
}

impl Indexer {
    /// Create new indexer
    pub fn new() -> Self {
        Self {
            use_unix_newline: false,
            tables: HashMap::new(),
        }
    }

    /// Always use unix newline for formatting
    pub fn always_use_unix_newline(&mut self, tv: bool) {
        self.use_unix_newline = tv;
    }

    /// Return newline with unix newline option considered
    fn get_newline(&self) -> &str {
        if self.use_unix_newline {
            "\n"
        } else {
            consts::LINE_ENDING
        }
    }

    /// Check if indexer contains table
    pub fn contains_table(&self, table_name: &str) -> bool {
        self.tables.contains_key(table_name)
    }

    /// Drop table
    pub fn drop_table(&mut self, table_name: &str) {
        self.tables.remove(table_name);
    }

    /// Add table without header
    pub fn add_table_fast(&mut self, table_name: &str, input: impl Read) -> CIndexResult<()> {
        self.add_table(table_name, vec![], input)
    }

    /// Add table
    pub fn add_table(
        &mut self,
        table_name: &str,
        header_types: Vec<CsvType>,
        mut input: impl Read,
    ) -> CIndexResult<()> {
        let mut table_content = String::new();
        input.read_to_string(&mut table_content)?;

        let mut lines = table_content.lines();
        let headers: Vec<(String, CsvType)>;
        let mut rows = vec![];

        if let Some(headers_line) = lines.next() {
            // Pad headers if heade's length is longer than header_types

            let header_types_iter = header_types[0..]
                .iter()
                .chain(std::iter::repeat(&CsvType::Text));
            let header_lines_iter = headers_line.split(',');

            // NOTE
            // Technically speaking, types can be bigger than header values length
            // But it yields expectable behaviour, thus it stays as it is.
            let len = header_lines_iter.clone().collect::<Vec<&str>>().len();

            headers = header_types_iter
                .zip(header_lines_iter)
                .take(len)
                .map(|(value, t)| (t.to_owned(), *value))
                .collect();
        } else {
            return Err(CIndexError::InvalidTableInput(format!(
                "No header option is not supported"
            )));
        }

        for line in lines {
            let row: Vec<&str> = line.split(',').collect();

            if row.len() != headers.len() {
                return Err(CIndexError::InvalidTableInput(format!(
                    "Row's length \"{}\" is different from header's length \"{}\"",
                    row.len(),
                    headers.len()
                )));
            }

            rows.push(row);
        }

        self.tables
            .insert(table_name.to_owned(), Table::new(headers, rows)?);
        Ok(())
    }

    //<INDEXING>
    /// Index with raq query
    pub fn index_raw(&self, raw_query: &str, out_option: OutOption) -> CIndexResult<()> {
        self.index(Query::from_str(raw_query)?, out_option)
    }

    /// Index with pre-built query
    pub fn index(&self, query: Query, mut out_option: OutOption) -> CIndexResult<()> {
        let records = self.index_table(query)?;

        for row in records {
            self.write(&(row.join(",") + self.get_newline()), &mut out_option)?;
        }
        Ok(())
    }

    /// Get rows filtered by query
    pub fn index_get_records(&self, query: Query) -> CIndexResult<Vec<Vec<String>>> {
        let records = self.index_table(query)?;
        Ok(records)
    }

    /// Internal function
    fn index_table(&self, query: Query) -> CIndexResult<Vec<Vec<String>>> {
        let mut output_header = vec![];
        let mut mapped_records: Vec<Vec<&str>> = vec![];
        let table =
            self.tables
                .get(query.table_name.as_str())
                .ok_or(CIndexError::InvalidTableName(format!(
                    "Table \"{}\" doesn't exist",
                    query.table_name
                )))?;
        let queried_records = table.query(&query)?;

        let target_columns: Option<Vec<ColumnIndex>> = if let Some(ref cols) = query.column_names {
            if cols.len() > 0 && cols[0] == "*" {
                None
            } else {
                #[cfg(feature = "rayon")]
                let iter = cols.par_iter();
                #[cfg(not(feature = "rayon"))]
                let iter = cols.iter();

                // If supplement is given
                // add extra columns
                let supplement = query.flags.contains(QueryFlags::SUP);
                let collection: Vec<_> = if supplement {
                    iter.map(|i| {
                        if let Some(index) = table.headers.get_index_of(i) {
                            ColumnIndex::Real(index)
                        } else {
                            ColumnIndex::Supplement
                        }
                    })
                    .collect()
                } else {
                    iter.map(|i| -> Result<ColumnIndex, CIndexError> {
                        let index = ColumnIndex::Real(table.headers.get_index_of(i).ok_or(
                            CIndexError::InvalidColumn(format!("No such column \"{}\"", i)),
                        )?);
                        Ok(index)
                    })
                    .collect::<CIndexResult<Vec<_>>>()?
                };

                Some(collection)
            }
        } else {
            None
        };

        // Print headers
        if query.flags.contains(QueryFlags::PHD) {
            let columns = query.column_names.unwrap_or(vec!["*".to_owned()]);
            let map = query.column_map.unwrap_or(vec![]);
            output_header = if columns[0] == "*" {
                let headers = table
                    .headers
                    .iter()
                    .map(|s| s.to_string())
                    .collect::<Vec<String>>();

                map.iter()
                    .chain(headers[map.len()..].iter())
                    .map(|s| s.to_string())
                    .collect::<Vec<String>>()
            } else {
                map.iter()
                    .chain(columns[map.len()..].iter())
                    .map(|s| s.to_string())
                    .collect::<Vec<String>>()
            };
            mapped_records.push(output_header.iter().map(|s| s.as_str()).collect());
        }

        let mut records_iter = queried_records.iter();

        while let Some(&row) = records_iter.next() {
            if let Some(cols) = &target_columns {
                mapped_records.push(row.column_splited(cols))
            } else {
                mapped_records.push(row.splited())
            };
        }

        // Tranpose if given TP Flag
        if query.flags.contains(QueryFlags::TP) {
            mapped_records = self.tranpose_records(mapped_records);
        }

        Ok(mapped_records
            .iter_mut()
            .map(|vec| vec.iter_mut().map(|val| val.to_string()).collect())
            .collect())
    }

    // Tranpose
    // https://stackoverflow.com/questions/64498617/how-to-transpose-a-vector-of-vectors-in-rust
    // Thank you stackoverflow ;)
    fn tranpose_records<'a>(&'a self, v: Vec<Vec<&'a str>>) -> Vec<Vec<&'a str>> {
        let len = v[0].len();
        let mut iters: Vec<_> = v.into_iter().map(|n| n.into_iter()).collect();
        (0..len)
            .map(|_| {
                iters
                    .iter_mut()
                    .map(|n| n.next().unwrap())
                    .collect::<Vec<&str>>()
            })
            .collect()
    }

    fn write(&self, content: &str, out_option: &mut OutOption) -> CIndexResult<()> {
        match out_option {
            OutOption::Term => std::io::stdout().write_all(content.as_bytes())?,
            OutOption::File(file) => file.write_all(content.as_bytes())?,
            OutOption::Value(target) => target.push_str(content),
        }
        Ok(())
    }
}

/// Ouput redirect option
pub enum OutOption<'a> {
    Term,
    Value(&'a mut String),
    File(File),
}
