// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: GPL-2.0-only

//! This crate is a library for communicating with [Chrony](https://chrony.tuxfamily.org)'s
//! control & monitoring interface. It provides programmatic access to information that you'd
//! otherwise have to scrape from the output of `chronyc`.
//!
//! For simple use cases which involve non-privileged queries to the locally-running Chrony
//! daemon, your main entry-point into the library will be the [blocking_query] function with
//! a `server` argument of `&LOCAL_SERVER_ADDR`. For privileged commands, use [blocking_query_uds]
//! instead (this will require permissions to write to the `/var/run/chrony` directory).
//!
//! ```no_run
//! use chrony_candm::request::RequestBody;
//! use chrony_candm::reply::ReplyBody;
//! use chrony_candm::{blocking_query,LOCAL_SERVER_ADDR};
//!
//! let request_body = RequestBody::Tracking;
//! let options = Default::default();
//! let reply = blocking_query(request_body, options, &LOCAL_SERVER_ADDR)?;
//! if let ReplyBody::Tracking(tracking) = reply.body {
//!     println!("The current RMS offset is {} seconds.", tracking.rms_offset);
//! }
//! # Ok::<(), std::io::Error>(())
//! ```
//!
//! Asynchronous applications can use [Client] instead of the standalone
//! `blocking_query` and `blocking_query_uds` functions.

extern crate self as chrony_candm; //We need this so that the fully qualified identifiers generated by macros can resolve

#[cfg(feature = "with_tokio")]
mod async_net;
pub mod common;
mod net;
pub mod reply;
pub mod request;

#[cfg(feature = "with_tokio")]
pub use async_net::*;
pub use net::*;
