use std::borrow::Borrow;
use argh::FromArgs;
use std::fs::File;
use std::io::{Error, stdin};
use std::path::{Path, PathBuf};
use std::io::prelude::*;
use std::num::Wrapping;
use std::process::exit;
use byteorder::{ByteOrder, LittleEndian};

pub enum Bits {
    U32,
    U8
}

pub fn default_bits() -> Bits {
    Bits::U32
}

pub fn bits_from_str(_value: &str) -> Result<Bits, String> {
    match _value {
        "32" => {return Ok(Bits::U32)},
        "8" => {return Ok(Bits::U8)},
        _ => {println!("Error: value of argument 'bits' must be 8 or 32."); exit(1)}
    }
}

#[derive(FromArgs)]
/// Calculates 8- and 32-bit little endian checksums.
/// If no paths are specified, checksum is calculated from stdin.
pub struct Args {
    /// select between 8 or 32 bit sum
    #[argh(option, from_str_fn(bits_from_str), default = "default_bits()", short = 'b')]
    pub bits: Bits,

    /// input file(s)
    #[argh(positional)]
    pub paths: Vec<PathBuf>,
}

pub fn run(args: Args) {
    if args.paths.is_empty() {
        let bytes = read_stdin().unwrap_or_else(|err| {
            println!("{}", err); exit(1);
        });

        match args.bits {
            Bits::U32 => {
                match checksum_32(bytes){
                    Ok(sum) => {println!("{:08X}  {}", sum, "-")}
                    Err(err) => {println!("Error: {}", err)}
                }
            },
            Bits::U8 => {
                match checksum_8(bytes){
                    Ok(sum) => {println!("{:02X} {}", sum, "-")}
                    Err(err) => {println!("Error: {}", err)}
                }
            }
        }

    } else {
        for path in args.paths {
            let pathb:&Path = path.borrow();
            let file_name = pathb.file_name().unwrap().to_str().unwrap();

            let bytes = read_file(pathb).unwrap_or_else(|err| {
                    println!("{}", err); exit(1);
                });

            match args.bits {
                Bits::U32 => {
                    match checksum_32(bytes){
                        Ok(sum) => {println!("{:010X}  {}", sum, file_name)}
                        Err(err) => {println!("Error: {}", err)}
                    }
                },
                Bits::U8 => {
                    match checksum_8(bytes){
                        Ok(sum) => {println!("{:04X} {}", sum, file_name)}
                        Err(err) => {println!("Error: {}", err)}
                    }
                }
            }
        }
    }
}

fn read_stdin() -> Result<Vec<u8>, Error> {
    let mut stdin = stdin();
    let mut buf = vec![];

    match stdin.read_to_end(&mut buf) {
        Ok(_) => {}
        Err(err) => return Err(err)
    }

    return Ok(buf)
}

fn read_file(filename: &Path) -> Result<Vec<u8>, Error> {
    let mut buf = vec![];

    {
        let mut file = File::open(filename).unwrap();
        match file.read_to_end(&mut buf) {
            Ok(_) => {}
            Err(err) => return Err(err)
        };
    }

    return Ok(buf)
}

fn checksum_8(bytes: Vec<u8>) -> Result<u32, &'static str> {
    let mut sum = Wrapping(0u8);

    for x in bytes.iter() {
        sum += x & 0xff;
    }

    return Ok(sum.0 as u32);
}

fn checksum_32(bytes: Vec<u8>) -> Result<u32, &'static str> {
    if bytes.len() % 4 != 0 {
        return Err("file length invalid, not a multiple of 4");
    }

    let mut sum = Wrapping(0u32);

    for x in (0..bytes.len()).step_by(4) {
        sum += LittleEndian::read_u32(&bytes[x..x+4]);
    }

    return Ok(sum.0);
}