pub const MEMORY_SIZE: usize = 4096;
pub const PROGRAM_START: usize = 0x200;

pub const FONTSET_ADDRESS: usize = 0x000;
const FONT_SET: [u8; 80] = [
    0xF0, 0x90, 0x90, 0x90, 0xF0, // 0
    0x20, 0x60, 0x20, 0x20, 0x70, // 1
    0xF0, 0x10, 0xF0, 0x80, 0xF0, // 2
    0xF0, 0x10, 0xF0, 0x10, 0xF0, // 3
    0x90, 0x90, 0xF0, 0x10, 0x10, // 4
    0xF0, 0x80, 0xF0, 0x10, 0xF0, // 5
    0xF0, 0x80, 0xF0, 0x90, 0xF0, // 6
    0xF0, 0x10, 0x20, 0x40, 0x40, // 7
    0xF0, 0x90, 0xF0, 0x90, 0xF0, // 8
    0xF0, 0x90, 0xF0, 0x10, 0xF0, // 9
    0xF0, 0x90, 0xF0, 0x90, 0x90, // A
    0xE0, 0x90, 0xE0, 0x90, 0xE0, // B
    0xF0, 0x80, 0x80, 0x80, 0xF0, // C
    0xE0, 0x90, 0x90, 0x90, 0xE0, // D
    0xF0, 0x80, 0xF0, 0x80, 0xF0, // E
    0xF0, 0x80, 0xF0, 0x80, 0x80, // F
];

#[derive(Clone, Debug)]
pub struct Memory {
    pub mem: Box<[u8]>,
}

impl Memory {
    pub fn new(program: &Vec<u8>) -> Result<Memory, &'static str> {
        let mut tmp = vec![0u8; MEMORY_SIZE].into_boxed_slice();

        load_fontset(&mut tmp);
        load_program(&mut tmp, program)?;

        Ok(Memory { mem: tmp })
    }
}

fn load_fontset(mem: &mut Box<[u8]>) {
    mem[FONTSET_ADDRESS..(FONTSET_ADDRESS + 80)].copy_from_slice(&FONT_SET);
}

fn load_program(mem: &mut Box<[u8]>, program: &Vec<u8>) -> Result<(), &'static str> {
    let last_address = PROGRAM_START + program.len();

    if last_address >= MEMORY_SIZE {
        return Err("Program cannot be stored in emulated RAM");
    }

    mem[PROGRAM_START..last_address].copy_from_slice(program);
    Ok(())
}
