#[test]
fn short() {
    use chinese_numerals::ShortScaleInt;

    let num = ShortScaleInt::try_from(-1_0023_0456_7890_i64).unwrap();
    assert_eq!(
        "负一涧零二秭三垓零四兆五亿六万七千八百九十",
        format!("{}", num)
    );
    assert_eq!(
        "负壹涧零贰秭叁垓零肆兆伍亿陆万柒仟捌佰玖拾",
        format!("{:#}", num)
    );

    assert_eq!("十三", format!("{}", ShortScaleInt::from(13)));
    assert_eq!("壹拾叁", format!("{:#}", ShortScaleInt::from(13)));
    assert_eq!("零", format!("{}", ShortScaleInt::default()));

    assert_eq!("负一百二十八", format!("{}", ShortScaleInt::from(i8::MIN)));
    assert_eq!("一百二十七", format!("{}", ShortScaleInt::from(i8::MAX)));
    assert_eq!("二百五十五", format!("{}", ShortScaleInt::from(u8::MAX)));
    assert_eq!(
        "负三万二千七百六十八",
        format!("{}", ShortScaleInt::from(i16::MIN))
    );
    assert_eq!(
        "三万二千七百六十七",
        format!("{}", ShortScaleInt::from(i16::MAX))
    );
    assert_eq!(
        "六万五千五百三十五",
        format!("{}", ShortScaleInt::from(u16::MAX))
    );
    assert_eq!(
        "负二秭一垓四京七兆四亿八万三千六百四十八",
        format!("{}", ShortScaleInt::from(i32::MIN))
    );
    assert_eq!(
        "二秭一垓四京七兆四亿八万三千六百四十七",
        format!("{}", ShortScaleInt::from(i32::MAX))
    );
    assert_eq!(
        "四秭二垓九京四兆九亿六万七千二百九十五",
        format!("{}", ShortScaleInt::from(u32::MAX))
    );

    assert_eq!(
        "九载九正九涧九沟九穰九秭九垓九京九兆九亿九万九千九百九十九",
        format!("{}", ShortScaleInt::MAX)
    );
    assert_eq!(
        "负九载九正九涧九沟九穰九秭九垓九京九兆九亿九万九千九百九十九",
        format!("{}", ShortScaleInt::MIN)
    );

    ShortScaleInt::try_from(1000_0000_0000_0000_i64).expect_err("Should be out of range");
}

#[test]
fn myriad() {
    use chinese_numerals::MyriadScaleInt;

    let num = MyriadScaleInt::from(-1_0023_0456_7890_i64);
    assert_eq!(
        "负一兆零二十三亿零四百五十六万七千八百九十",
        format!("{}", num)
    );
    assert_eq!(
        "负壹兆零贰拾叁亿零肆佰伍拾陆万柒仟捌佰玖拾",
        format!("{:#}", num)
    );

    assert_eq!(
        "十六万零四百二十",
        format!("{}", MyriadScaleInt::from(16_0420))
    );
    assert_eq!(
        "壹拾陆万零肆佰贰拾",
        format!("{:#}", MyriadScaleInt::from(16_0420))
    );
    assert_eq!("零", format!("{}", MyriadScaleInt::default()));

    assert_eq!(
        "负三万二千七百六十八",
        format!("{}", MyriadScaleInt::from(i16::MIN))
    );
    assert_eq!(
        "三万二千七百六十七",
        format!("{}", MyriadScaleInt::from(i16::MAX))
    );
    assert_eq!(
        "六万五千五百三十五",
        format!("{}", MyriadScaleInt::from(u16::MAX))
    );
    assert_eq!(
        "负二十一亿四千七百四十八万三千六百四十八",
        format!("{}", MyriadScaleInt::from(i32::MIN))
    );
    assert_eq!(
        "二十一亿四千七百四十八万三千六百四十七",
        format!("{}", MyriadScaleInt::from(i32::MAX))
    );
    assert_eq!(
        "四十二亿九千四百九十六万七千二百九十五",
        format!("{}", MyriadScaleInt::from(u32::MAX))
    );
    assert_eq!(
        "负九百二十二京三千三百七十二兆零三百六十八亿五千四百七十七万五千八百零八",
        format!("{}", MyriadScaleInt::from(i64::MIN))
    );
    assert_eq!(
        "九百二十二京三千三百七十二兆零三百六十八亿五千四百七十七万五千八百零七",
        format!("{}", MyriadScaleInt::from(i64::MAX))
    );
    assert_eq!(
        "一千八百四十四京六千七百四十四兆零七百三十七亿零九百五十五万一千六百一十五",
        format!("{}", MyriadScaleInt::from(u64::MAX))
    );
    assert_eq!(
        "负一百七十涧一千四百一十一沟八千三百四十六穰零四百六十九秭二千三百一十七垓\
        三千一百六十八京七千三百零三兆七千一百五十八亿八千四百一十万五千七百二十八",
        format!("{}", MyriadScaleInt::from(i128::MIN))
    );
    assert_eq!(
        "一百七十涧一千四百一十一沟八千三百四十六穰零四百六十九秭二千三百一十七垓\
        三千一百六十八京七千三百零三兆七千一百五十八亿八千四百一十万五千七百二十七",
        format!("{}", MyriadScaleInt::from(i128::MAX))
    );
    assert_eq!(
        "三百四十涧二千八百二十三沟六千六百九十二穰零九百三十八秭四千六百三十四垓\
        六千三百三十七京四千六百零七兆四千三百一十七亿六千八百二十一万一千四百五十五",
        format!("{}", MyriadScaleInt::from(u128::MAX))
    );
}

#[cfg(feature = "bigint")]
#[test]
fn big_myriad() {
    use chinese_numerals::MyriadScaleBigInt;
    use num_bigint::{BigInt, Sign};

    // -1_0023_0456_7890_0987_6540_3200_1001_0023_0456_7890_0000
    let num = BigInt::new(
        Sign::Minus,
        vec![3140100896, 3348699770, 1853619678, 3604935901, 294550],
    );
    let num = MyriadScaleBigInt::try_from(num).expect("Out of range");
    assert_eq!(
        "负一载零二十三正零四百五十六涧七千八百九十沟零九百八十七穰六千五百四十秭\
        三千二百垓一千零一京零二十三兆零四百五十六亿七千八百九十万",
        format!("{}", num)
    );
    assert_eq!(
        "负壹载零贰拾叁正零肆佰伍拾陆涧柒仟捌佰玖拾沟零玖佰捌拾柒穰陆仟伍佰肆拾秭\
        叁仟贰佰垓壹仟零壹京零贰拾叁兆零肆佰伍拾陆亿柒仟捌佰玖拾万",
        format!("{:#}", num)
    );

    assert_eq!("零", format!("{}", MyriadScaleBigInt::default()));

    // 9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999
    assert_eq!(
        "九千九百九十九载九千九百九十九正九千九百九十九涧九千九百九十九沟\
        九千九百九十九穰九千九百九十九秭九千九百九十九垓九千九百九十九京\
        九千九百九十九兆九千九百九十九亿九千九百九十九万九千九百九十九",
        format!("{}", MyriadScaleBigInt::max_value())
    );
    assert_eq!(
        "负九千九百九十九载九千九百九十九正九千九百九十九涧九千九百九十九沟\
        九千九百九十九穰九千九百九十九秭九千九百九十九垓九千九百九十九京\
        九千九百九十九兆九千九百九十九亿九千九百九十九万九千九百九十九",
        format!("{}", MyriadScaleBigInt::min_value())
    );

    // MyriadScaleBigInt::max_value() + 1
    let num = BigInt::new(
        Sign::Minus,
        vec![0, 2134966272, 2523967787, 239310294, 2938735877],
    );
    MyriadScaleBigInt::try_from(num).expect_err("Should be out of range");
}

#[test]
fn mid() {
    use chinese_numerals::MidScaleInt;

    let num = MidScaleInt::from(-10_0023_0456_7890_i64);
    assert_eq!(
        "负十万零二十三亿零四百五十六万七千八百九十",
        format!("{}", num)
    );
    assert_eq!(
        "负壹拾万零贰拾叁亿零肆佰伍拾陆万柒仟捌佰玖拾",
        format!("{:#}", num)
    );

    assert_eq!("零", format!("{}", MidScaleInt::default()));

    assert_eq!(
        "负三万二千七百六十八",
        format!("{}", MidScaleInt::from(i16::MIN))
    );
    assert_eq!(
        "三万二千七百六十七",
        format!("{}", MidScaleInt::from(i16::MAX))
    );
    assert_eq!(
        "六万五千五百三十五",
        format!("{}", MidScaleInt::from(u16::MAX))
    );
    assert_eq!(
        "负二十一亿四千七百四十八万三千六百四十八",
        format!("{}", MidScaleInt::from(i32::MIN))
    );
    assert_eq!(
        "二十一亿四千七百四十八万三千六百四十七",
        format!("{}", MidScaleInt::from(i32::MAX))
    );
    assert_eq!(
        "四十二亿九千四百九十六万七千二百九十五",
        format!("{}", MidScaleInt::from(u32::MAX))
    );
    assert_eq!(
        "负九百二十二兆三千三百七十二万零三百六十八亿五千四百七十七万五千八百零八",
        format!("{}", MidScaleInt::from(i64::MIN))
    );
    assert_eq!(
        "九百二十二兆三千三百七十二万零三百六十八亿五千四百七十七万五千八百零七",
        format!("{}", MidScaleInt::from(i64::MAX))
    );
    assert_eq!(
        "一千八百四十四兆六千七百四十四万零七百三十七亿零九百五十五万一千六百一十五",
        format!("{}", MidScaleInt::from(u64::MAX))
    );
    assert_eq!(
        "负一百七十万一千四百一十一垓八千三百四十六万零四百六十九京二千三百一十七万\
        三千一百六十八兆七千三百零三万七千一百五十八亿八千四百一十万五千七百二十八",
        format!("{}", MidScaleInt::from(i128::MIN))
    );
    assert_eq!(
        "一百七十万一千四百一十一垓八千三百四十六万零四百六十九京二千三百一十七万\
        三千一百六十八兆七千三百零三万七千一百五十八亿八千四百一十万五千七百二十七",
        format!("{}", MidScaleInt::from(i128::MAX))
    );
    assert_eq!(
        "三百四十万二千八百二十三垓六千六百九十二万零九百三十八京四千六百三十四万\
        六千三百三十七兆四千六百零七万四千三百一十七亿六千八百二十一万一千四百五十五",
        format!("{}", MidScaleInt::from(u128::MAX))
    );
}

#[cfg(feature = "bigint")]
#[test]
fn big_mid() {
    use chinese_numerals::MidScaleBigInt;
    use num_bigint::{BigInt, Sign};

    // -1_0023_0456_7890_0987_6540_3200_1001_0023_0456_7890_0000
    let num = BigInt::new(
        Sign::Minus,
        vec![3140100896, 3348699770, 1853619678, 3604935901, 294550],
    );
    let num = MidScaleBigInt::try_from(num).expect("Out of range");
    assert_eq!(
        "负一万零二十三秭零四百五十六万七千八百九十垓零九百八十七万六千五百四十京\
        三千二百万一千零一兆零二十三万零四百五十六亿七千八百九十万",
        format!("{}", num)
    );
    assert_eq!(
        "负壹万零贰拾叁秭零肆佰伍拾陆万柒仟捌佰玖拾垓零玖佰捌拾柒万陆仟伍佰肆拾京\
        叁仟贰佰万壹仟零壹兆零贰拾叁万零肆佰伍拾陆亿柒仟捌佰玖拾万",
        format!("{:#}", num)
    );

    assert_eq!("零", format!("{}", MidScaleBigInt::default()));

    // 9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999\
    // 9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999
    assert_eq!(
        "九千九百九十九万九千九百九十九载九千九百九十九万九千九百九十九正九千九百九十九万\
        九千九百九十九涧九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九穰\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九",
        format!("{}", MidScaleBigInt::max_value())
    );
    assert_eq!(
        "负九千九百九十九万九千九百九十九载九千九百九十九万九千九百九十九正九千九百九十九万\
        九千九百九十九涧九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九穰\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九",
        format!("{}", MidScaleBigInt::min_value())
    );

    // MidScaleBigInt::max_value() + 1
    let num = BigInt::new(
        Sign::Minus,
        vec![
            0, 0, 2701131776, 807615852, 3882706566, 3057181734, 745289159, 4056365773, 462339630,
            20,
        ],
    );
    MidScaleBigInt::try_from(num).expect_err("Should be out of range");
}

#[test]
fn long() {
    use chinese_numerals::LongScaleInt;

    let num = LongScaleInt::from(-10_0023_0456_7890_9876_5432_1023_i128);
    assert_eq!(
        "负十亿零二十三万零四百五十六兆七千八百九十万九千八百七十六亿五千四百三十二万一千零二十三",
        format!("{}", num)
    );
    assert_eq!(
        "负壹拾亿零贰拾叁万零肆佰伍拾陆兆柒仟捌佰玖拾万玖仟捌佰柒拾陆亿伍仟肆佰叁拾贰万壹仟零贰拾叁",
        format!("{:#}", num)
    );

    assert_eq!("零", format!("{}", LongScaleInt::default()));

    assert_eq!(
        "负三万二千七百六十八",
        format!("{}", LongScaleInt::from(i16::MIN))
    );
    assert_eq!(
        "三万二千七百六十七",
        format!("{}", LongScaleInt::from(i16::MAX))
    );
    assert_eq!(
        "六万五千五百三十五",
        format!("{}", LongScaleInt::from(u16::MAX))
    );
    assert_eq!(
        "负二十一亿四千七百四十八万三千六百四十八",
        format!("{}", LongScaleInt::from(i32::MIN))
    );
    assert_eq!(
        "二十一亿四千七百四十八万三千六百四十七",
        format!("{}", LongScaleInt::from(i32::MAX))
    );
    assert_eq!(
        "四十二亿九千四百九十六万七千二百九十五",
        format!("{}", LongScaleInt::from(u32::MAX))
    );
    assert_eq!(
        "负九百二十二兆三千三百七十二万零三百六十八亿五千四百七十七万五千八百零八",
        format!("{}", LongScaleInt::from(i64::MIN))
    );
    assert_eq!(
        "九百二十二兆三千三百七十二万零三百六十八亿五千四百七十七万五千八百零七",
        format!("{}", LongScaleInt::from(i64::MAX))
    );
    assert_eq!(
        "一千八百四十四兆六千七百四十四万零七百三十七亿零九百五十五万一千六百一十五",
        format!("{}", LongScaleInt::from(u64::MAX))
    );
    assert_eq!(
        "负一百七十万一千四百一十一京八千三百四十六万零四百六十九亿二千三百一十七万\
        三千一百六十八兆七千三百零三万七千一百五十八亿八千四百一十万五千七百二十八",
        format!("{}", LongScaleInt::from(i128::MIN))
    );
    assert_eq!(
        "一百七十万一千四百一十一京八千三百四十六万零四百六十九亿二千三百一十七万\
        三千一百六十八兆七千三百零三万七千一百五十八亿八千四百一十万五千七百二十七",
        format!("{}", LongScaleInt::from(i128::MAX))
    );
    assert_eq!(
        "三百四十万二千八百二十三京六千六百九十二万零九百三十八亿四千六百三十四万\
        六千三百三十七兆四千六百零七万四千三百一十七亿六千八百二十一万一千四百五十五",
        format!("{}", LongScaleInt::from(u128::MAX))
    );
}

#[cfg(feature = "bigint")]
#[test]
fn big_long() {
    use chinese_numerals::LongScaleBigInt;
    use num_bigint::{BigInt, Sign};

    // -1_0023_0456_7890_0987_6540_3200_1001_0023_0456_7890_0000
    let num = BigInt::new(
        Sign::Minus,
        vec![3140100896, 3348699770, 1853619678, 3604935901, 294550],
    );
    let num = LongScaleBigInt::try_from(num).expect("Out of range");
    assert_eq!(
        "负一万零二十三亿零四百五十六万七千八百九十京零九百八十七万六千五百四十亿\
        三千二百万一千零一兆零二十三万零四百五十六亿七千八百九十万",
        format!("{}", num)
    );
    assert_eq!(
        "负壹万零贰拾叁亿零肆佰伍拾陆万柒仟捌佰玖拾京零玖佰捌拾柒万陆仟伍佰肆拾亿\
        叁仟贰佰万壹仟零壹兆零贰拾叁万零肆佰伍拾陆亿柒仟捌佰玖拾万",
        format!("{:#}", num)
    );

    assert_eq!("零", format!("{}", LongScaleBigInt::default()));

    assert_eq!(
        "九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九涧九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九正九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九涧九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九载九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九涧\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九正九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九涧九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿\
        九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万\
        九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京\
        九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万\
        九千九百九十九亿九千九百九十九万九千九百九十九",
        format!("{}", LongScaleBigInt::max_value())
    );
}
