mod macros;
mod characters;
mod shortscale;
mod myriadscale;
mod midscale;
mod longscale;

use characters::NumChar;
pub use shortscale::ShortScaleInt;
pub use myriadscale::MyriadScaleInt;
pub use midscale::MidScaleInt;
pub use longscale::LongScaleInt;

#[cfg(feature = "bigint")]
use num_bigint::BigUint;

#[cfg(feature = "bigint")]
pub use myriadscale::MyriadScaleBigInt;
#[cfg(feature = "bigint")]
pub use midscale::MidScaleBigInt;
#[cfg(feature = "bigint")]
pub use longscale::LongScaleBigInt;

/// The signs of the number.
#[derive(PartialEq, Eq, PartialOrd, Ord, Clone, Copy, Debug)]
pub enum Sign {
    Neg,
    Nil,
    Pos,
}

/// Chinese variants. For simplified and traditional.
#[derive(PartialEq, Eq, Clone, Copy, Debug)]
pub enum Variant {
    Simplified,
    Traditional,
}

impl Default for Sign {
    fn default() -> Self {
        Self::Nil
    }
}

/// Out of range errors.
#[cfg(feature = "bigint")]
#[derive(Debug)]
pub enum Error {
    ShortScaleOutOfRange(u128),
    MyriadScaleOutOfRange(BigUint),
    MidScaleOutOfRange(BigUint),
    LongScaleOutOfRange(BigUint),
}

#[cfg(feature = "bigint")]
impl std::fmt::Display for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Error::ShortScaleOutOfRange(value) => write!(f, "Absolute value {value} out of range for a short scale number"),
            Error::MyriadScaleOutOfRange(value) => write!(f, "Absolute value {value} out of range for a myriad scale number"),
            Error::MidScaleOutOfRange(value) => write!(f, "Absolute value {value} out of range for a mid-scale number"),
            Error::LongScaleOutOfRange(value) => write!(f, "Absolute value {value} out of range for a long scale number"),
        }
    }
}

#[cfg(not(feature = "bigint"))]
#[derive(Debug)]
pub enum Error {
    ShortScaleOutOfRange(u128),
}

#[cfg(not(feature = "bigint"))]
impl std::fmt::Display for Error {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match *self {
            Error::ShortScaleOutOfRange(value) => write!(f, "Absolute value {value} out of range for a short scale number"),
        }
    }
}

impl std::error::Error for Error {}

/// Provides methods to generate Chinease numeral expression for a number.
pub trait ChineseNumeral {
    type Data;

    fn sign(&self) -> Sign;

    fn data(&self) -> &Self::Data;

    /// Generates the vector containing Chinese numeral characters for the number, in reverse order.
    fn to_chars(&self) -> Vec<NumChar>;

    /// Does similar thing with `to_chars`, but with 「一」 in leading 「一十」 trimmed.
    fn to_chars_trimmed(&self) -> Vec<NumChar>;

    /// Converts the number to lowercase (小写数字, used for normal writing).
    fn to_lowercase(&self, variant: Variant) -> String {
        let method = match variant {
            Variant::Simplified => NumChar::to_lowercase_simp,
            Variant::Traditional => NumChar::to_lowercase_trad,
        };
        let mut chars = self.to_chars_trimmed();
        match self.sign() {
            Sign::Neg => chars.push(NumChar::Neg),
            Sign::Nil => chars.push(NumChar::Zero),
            _ => {},
        }
        chars.into_iter()
            .rev()
            .map(method)
            .collect()
    }

    /// Converts the number to lowercase (小写数字, used for normal writing) in simplified Chinese.
    fn to_lowercase_simp(&self) -> String {
        self.to_lowercase(Variant::Simplified)
    }

    /// Converts the number to lowercase (小写数字, used for normal writing) in traditional Chinese.
    fn to_lowercase_trad(&self) -> String {
        self.to_lowercase(Variant::Traditional)
    }

    /// Converts the number to uppercase (大写数字, used for financial contexts).
    fn to_uppercase(&self, variant: Variant) -> String {
        let method = match variant {
            Variant::Simplified => NumChar::to_uppercase_simp,
            Variant::Traditional => NumChar::to_uppercase_trad,
        };
        let mut chars = self.to_chars();
        match self.sign() {
            Sign::Neg => chars.push(NumChar::Neg),
            Sign::Nil => chars.push(NumChar::Zero),
            _ => {},
        }
        chars.into_iter()
            .rev()
            .map(method)
            .collect()
    }

    /// Converts the number to uppercase (大写数字, used for financial contexts) in simplified Chinese.
    fn to_uppercase_simp(&self) -> String {
        self.to_uppercase(Variant::Simplified)
    }

    /// Converts the number to uppercase (大写数字, used for financial contexts) in traditional Chinese.
    fn to_uppercase_trad(&self) -> String {
        self.to_uppercase(Variant::Traditional)
    }
}

#[cfg(test)]
mod tests {
    use crate::{ShortScaleInt, MyriadScaleInt, Sign, MidScaleInt, LongScaleInt};

    #[test]
    fn short() {
        let num = ShortScaleInt::from(1_0203_0405);
        assert_eq!("一垓零二兆零三万零四百零五", format!("{}", num));
        assert_eq!("壹垓零贰兆零叁万零肆佰零伍", format!("{:#}", num));

        let num = ShortScaleInt::from(13);
        assert_eq!("十三", format!("{}", num));

        let zero = ShortScaleInt::default();
        assert_eq!("零", format!("{}", zero));

        // 999_9999_9999_9999
        let max = ShortScaleInt::MAX;
        assert_eq!("九载九正九涧九沟九穰九秭九垓九京九兆九亿九万九千九百九十九", format!("{}", max));

        let min = ShortScaleInt::MIN;
        assert_eq!("负九载九正九涧九沟九穰九秭九垓九京九兆九亿九万九千九百九十九", format!("{}", min));

        assert!(ShortScaleInt::try_from(1000_0000_0000_0000_i64).is_err());
    }

    #[test]
    fn myriad() {
        let num = MyriadScaleInt::from(2_1239_4853_3777_i64);
        assert_eq!("二兆一千二百三十九亿四千八百五十三万三千七百七十七", format!("{}", num));

        let zero = MyriadScaleInt::default();
        assert_eq!("零", format!("{}", zero));

        // 340_2823_6692_0938_4634_6337_4607_4317_6821_1455
        let max = MyriadScaleInt::from(u128::MAX);
        assert_eq!("三百四十涧二千八百二十三沟六千六百九十二穰零九百三十八秭四千六百三十四垓六千三百三十七京四千六百零七兆四千三百一十七亿六千八百二十一万一千四百五十五", format!("{}", max));

        let min = MyriadScaleInt::new(Sign::Neg, u128::MAX);
        assert_eq!("负三百四十涧二千八百二十三沟六千六百九十二穰零九百三十八秭四千六百三十四垓六千三百三十七京四千六百零七兆四千三百一十七亿六千八百二十一万一千四百五十五", format!("{}", min));
    }

    #[test]
    fn mid() {
        let num = MidScaleInt::from(-19_7012_1011_0011_0000_0011_i128);
        assert_eq!("负十九万七千零一十二兆一千零一十一万零一十一亿零一十一", format!("{}", num));

        let zero = MidScaleInt::default();
        assert_eq!("零", format!("{}", zero));

        // 340_2823_6692_0938_4634_6337_4607_4317_6821_1455
        let max = MidScaleInt::from(u128::MAX);
        assert_eq!("三百四十万二千八百二十三垓六千六百九十二万零九百三十八京四千六百三十四万六千三百三十七兆四千六百零七万四千三百一十七亿六千八百二十一万一千四百五十五", format!("{}", max));

        // -170_1411_8346_0469_2317_3168_7303_7158_8410_5728
        let neg = MidScaleInt::from(i128::MIN);
        assert_eq!("负一百七十万一千四百一十一垓八千三百四十六万零四百六十九京二千三百一十七万三千一百六十八兆七千三百零三万七千一百五十八亿八千四百一十万五千七百二十八", format!("{}", neg));
    }

    #[test]
    fn long() {
        let num = LongScaleInt::from(-10_0831_1936_0193_6621_9381_i128);
        // 10_0831_1936_0193_6621_9381
        assert_eq!("负十万零八百三十一兆一千九百三十六万零一百九十三亿六千六百二十一万九千三百八十一", format!("{}", num));

        let zero = LongScaleInt::default();
        assert_eq!("零", format!("{}", zero));

        // 340_2823_6692_0938_4634_6337_4607_4317_6821_1455
        let max = LongScaleInt::from(u128::MAX);
        assert_eq!("三百四十万二千八百二十三京六千六百九十二万零九百三十八亿四千六百三十四万六千三百三十七兆四千六百零七万四千三百一十七亿六千八百二十一万一千四百五十五", format!("{}", max));
    }

    #[cfg(feature = "bigint")]
    #[test]
    fn myriad_bigint() {
        use crate::MyriadScaleBigInt;

        // 9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999
        let max = MyriadScaleBigInt::max_value();
        assert_eq!("九千九百九十九载九千九百九十九正九千九百九十九涧九千九百九十九沟九千九百九十九穰九千九百九十九秭九千九百九十九垓九千九百九十九京九千九百九十九兆九千九百九十九亿九千九百九十九万九千九百九十九", format!("{}", max));
    }

    #[cfg(feature = "bigint")]
    #[test]
    fn mid_bigint() {
        use crate::MidScaleBigInt;

        // 9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999_9999
        let max = MidScaleBigInt::max_value();
        assert_eq!("九千九百九十九万九千九百九十九载九千九百九十九万九千九百九十九正九千九百九十九万九千九百九十九涧九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九", format!("{}", max));
    }

    #[cfg(feature = "bigint")]
    #[test]
    fn long_bigint() {
        use num_bigint::BigUint;

        use crate::LongScaleBigInt;

        let ubig = BigUint::new(vec![10]).pow(4096);
        let num = LongScaleBigInt::try_from(ubig);
        assert!(num.is_ok());
        assert_eq!("一载", format!("{}", num.unwrap()));

        let max = LongScaleBigInt::max_value();
        assert_eq!("九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九涧九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九正九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九涧九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九载九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九涧九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九正九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九涧九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九沟九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九穰九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九秭九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九垓九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九京九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九兆九千九百九十九万九千九百九十九亿九千九百九十九万九千九百九十九", format!("{}", max));
    }
}
