#![cfg(test)]

use super::*;

#[test]
fn test_digit_1() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Upper);

    for i in 0..=9 {
        digit_1(chinese_number_table, i, &mut s);
    }

    assert_eq!("零壹貳參肆伍陸柒捌玖", s);
}

#[test]
fn test_digit_10_independently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Upper);

    digit_10(chinese_number_table, ChineseNumberCase::Upper, false, 10, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, false, 11, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, false, 12, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, false, 30, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, false, 95, &mut s);

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10(chinese_number_table, ChineseNumberCase::Lower, false, 10, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, false, 11, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, false, 12, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, false, 30, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, false, 95, &mut s);

    assert_eq!("壹拾壹拾壹壹拾貳參拾玖拾伍十十一十二三十九十五", s);
}

#[test]
fn test_digit_10_dependently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Upper);

    digit_10(chinese_number_table, ChineseNumberCase::Upper, true, 10, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, true, 11, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, true, 12, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, true, 30, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Upper, true, 95, &mut s);

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10(chinese_number_table, ChineseNumberCase::Lower, true, 10, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, true, 11, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, true, 12, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, true, 30, &mut s);
    digit_10(chinese_number_table, ChineseNumberCase::Lower, true, 95, &mut s);

    assert_eq!("壹拾壹拾壹壹拾貳參拾玖拾伍一十一十一一十二三十九十五", s);
}

#[test]
fn test_digit_100() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Upper);

    digit_100(chinese_number_table, ChineseNumberCase::Upper, 100, &mut s);
    digit_100(chinese_number_table, ChineseNumberCase::Upper, 101, &mut s);
    digit_100(chinese_number_table, ChineseNumberCase::Upper, 102, &mut s);
    digit_100(chinese_number_table, ChineseNumberCase::Upper, 110, &mut s);
    digit_100(chinese_number_table, ChineseNumberCase::Upper, 111, &mut s);
    digit_100(chinese_number_table, ChineseNumberCase::Upper, 200, &mut s);
    digit_100(chinese_number_table, ChineseNumberCase::Upper, 950, &mut s);
    digit_100(chinese_number_table, ChineseNumberCase::Upper, 999, &mut s);

    assert_eq!("壹佰壹佰零壹壹佰零貳壹佰壹拾壹佰壹拾壹貳佰玖佰伍拾玖佰玖拾玖", s);
}

#[test]
fn test_digit_1000() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Upper);

    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1000, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1001, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1010, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1011, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1100, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1101, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1110, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 1111, &mut s);
    digit_1000(chinese_number_table, ChineseNumberCase::Upper, 9999, &mut s);

    assert_eq!("壹仟壹仟零壹壹仟零壹拾壹仟零壹拾壹壹仟壹佰壹仟壹佰零壹壹仟壹佰壹拾壹仟壹佰壹拾壹玖仟玖佰玖拾玖", s);
}

#[test]
fn test_digit_10_000_independently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Upper);

    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, false, 10000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, false, 10001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, false, 100001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, false, 110010, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, false, 1001000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, false, 1100101, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, false, 99999999, &mut s);

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, false, 10000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, false, 10001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, false, 100001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, false, 110010, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, false, 1001000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, false, 1100101, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, false, 99999999, &mut s);

    assert_eq!("壹萬壹萬零壹壹拾萬零壹壹拾壹萬零壹拾壹佰萬壹仟壹佰壹拾萬零壹佰零壹玖仟玖佰玖拾玖萬玖仟玖佰玖拾玖一萬一萬零一十萬零一十一萬零一十一百萬一千一百一十萬零一百零一九千九百九十九萬九千九百九十九", s);
}

#[test]
fn test_digit_10_000_dependently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Upper);

    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, true, 10000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, true, 10001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, true, 100001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, true, 110010, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, true, 1001000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, true, 1100101, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Upper, true, 99999999, &mut s);

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, true, 10000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, true, 10001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, true, 100001, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, true, 110010, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, true, 1001000, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, true, 1100101, &mut s);
    digit_10_000(chinese_number_table, ChineseNumberCase::Lower, true, 99999999, &mut s);

    assert_eq!("壹萬壹萬零壹壹拾萬零壹壹拾壹萬零壹拾壹佰萬壹仟壹佰壹拾萬零壹佰零壹玖仟玖佰玖拾玖萬玖仟玖佰玖拾玖一萬一萬零一一十萬零一一十一萬零一十一百萬一千一百一十萬零一百零一九千九百九十九萬九千九百九十九", s);
}

#[test]
fn test_digit_1000_compat_independently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 0, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 1, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 10, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 22, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 333, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 4444, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 9090, &mut s);

    assert_eq!("零一十二十二三百三十三四千四百四十四九千零九十", s);
}

#[test]
fn test_digit_1000_compat_dependently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 0, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 1, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 10, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 22, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 333, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 4444, &mut s);
    digit_1000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 9090, &mut s);

    assert_eq!("一一十二十二三百三十三四千四百四十四九千零九十", s);
}

#[test]
fn test_digit_10_000_compat_independently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 0, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 1, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 10, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 22, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 333, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 4444, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 55555, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 100000, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 666066, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 990909, &mut s);

    assert_eq!("零一十二十二三百三十三四千四百四十四五萬五千五百五十五十萬六十六萬六千零六十六九十九萬零九百零九", s);
}

#[test]
fn test_digit_10_000_compat_dependently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 0, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 1, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 10, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 22, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 333, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 4444, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 55555, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 100000, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 666066, &mut s);
    digit_10_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 990909, &mut s);

    assert_eq!("一一十二十二三百三十三四千四百四十四五萬五千五百五十五一十萬六十六萬六千零六十六九十九萬零九百零九", s);
}

#[test]
fn test_digit_100_000_000_compat_independently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 0, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 1, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 10, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 2222, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, false, 333333, &mut s);
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        44444444,
        &mut s,
    );
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        5555555555,
        &mut s,
    );
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        1000000000000000,
        &mut s,
    );
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        9990099009900909,
        &mut s,
    );

    assert_eq!("零一十二千二百二十二三十三萬三千三百三十三四千四百四十四萬四千四百四十四五十五億五千五百五十五萬五千五百五十五一千萬億九千九百九十萬零九百九十億零九百九十萬零九百零九", s);
}

#[test]
fn test_digit_100_000_000_compat_dependently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 0, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 1, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 10, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 2222, &mut s);
    digit_100_000_000_compat(chinese_number_table, ChineseNumberCase::Lower, true, 333333, &mut s);
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        44444444,
        &mut s,
    );
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        5555555555,
        &mut s,
    );
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        1000000000000000,
        &mut s,
    );
    digit_100_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        9990099009900909,
        &mut s,
    );

    assert_eq!("一一十二千二百二十二三十三萬三千三百三十三四千四百四十四萬四千四百四十四五十五億五千五百五十五萬五千五百五十五一千萬億九千九百九十萬零九百九十億零九百九十萬零九百零九", s);
}

#[test]
fn test_digit_10_000_000_000_000_000_compat_independently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        0,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        1,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        10,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        222222,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        3333333333,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        44444444444444,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        555555555555555555,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        1000000000000000000000000000,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        9999900009999000099990000909,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        10000000000000000000000000000000,
        &mut s,
    );

    assert_eq!("零一十二十二萬二千二百二十二三十三億三千三百三十三萬三千三百三十三四十四萬四千四百四十四億四千四百四十四萬四千四百四十四五十五兆五千五百五十五萬五千五百五十五億五千五百五十五萬五千五百五十五一千億兆九千九百九十九億九千萬零九百九十九兆九千萬零九百九十九億九千萬零九百零九一千萬億兆", s);
}

#[test]
fn test_digit_10_000_000_000_000_000_compat_dependently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        0,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        1,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        10,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        222222,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        3333333333,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        44444444444444,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        555555555555555555,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        1000000000000000000000000000,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        9999900009999000099990000909,
        &mut s,
    );
    digit_10_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        10000000000000000000000000000000,
        &mut s,
    );

    assert_eq!("一一十二十二萬二千二百二十二三十三億三千三百三十三萬三千三百三十三四十四萬四千四百四十四億四千四百四十四萬四千四百四十四五十五兆五千五百五十五萬五千五百五十五億五千五百五十五萬五千五百五十五一千億兆九千九百九十九億九千萬零九百九十九兆九千萬零九百九十九億九千萬零九百零九一千萬億兆", s);
}

#[test]
fn test_digit_100_000_000_000_000_000_000_000_000_000_000_compat_independently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        0,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        1,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        10,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        22222222,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        33333333333333,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        44444444444444444444,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        55555555555555555555555555,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        100000000000000000000000000000000000000,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        false,
        333333000000333333000000333333000000303,
        &mut s,
    );

    assert_eq!("零一十二千二百二十二萬二千二百二十二三十三萬三千三百三十三億三千三百三十三萬三千三百三十三四千四百四十四兆四千四百四十四萬四千四百四十四億四千四百四十四萬四千四百四十四五十五億五千五百五十五萬五千五百五十五兆五千五百五十五萬五千五百五十五億五千五百五十五萬五千五百五十五一百萬京三百三十三萬三千三百三十京零三百三十三億三千三百三十萬兆零三百三十三萬三千三百三十億零三百零三", s);
}

#[test]
fn test_digit_100_000_000_000_000_000_000_000_000_000_000_compat_dependently() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        0,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        1,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        10,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        22222222,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        33333333333333,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        44444444444444444444,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        55555555555555555555555555,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        100000000000000000000000000000000000000,
        &mut s,
    );
    digit_100_000_000_000_000_000_000_000_000_000_000_compat(
        chinese_number_table,
        ChineseNumberCase::Lower,
        true,
        333333000000333333000000333333000000303,
        &mut s,
    );

    assert_eq!("一一十二千二百二十二萬二千二百二十二三十三萬三千三百三十三億三千三百三十三萬三千三百三十三四千四百四十四兆四千四百四十四萬四千四百四十四億四千四百四十四萬四千四百四十四五十五億五千五百五十五萬五千五百五十五兆五千五百五十五萬五千五百五十五億五千五百五十五萬五千五百五十五一百萬京三百三十三萬三千三百三十京零三百三十三億三千三百三十萬兆零三百三十三萬三千三百三十億零三百零三", s);
}

#[test]
fn test_digit_compat_low_u32() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_compat_low_u32(chinese_number_table, ChineseNumberCase::Lower, 0, &mut s);
    digit_compat_low_u32(chinese_number_table, ChineseNumberCase::Lower, 1, &mut s);
    digit_compat_low_u32(chinese_number_table, ChineseNumberCase::Lower, 10, &mut s);
    digit_compat_low_u32(chinese_number_table, ChineseNumberCase::Lower, 99999, &mut s);
    digit_compat_low_u32(chinese_number_table, ChineseNumberCase::Lower, 100000, &mut s);
    digit_compat_low_u32(chinese_number_table, ChineseNumberCase::Lower, u32::max_value(), &mut s);

    assert_eq!("零一十九萬九千九百九十九一億四秭二垓九京四兆九億六萬七千二百九十五", s);
}

#[test]
fn test_digit_compat_low_u64() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 0, &mut s);
    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 1, &mut s);
    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 10, &mut s);
    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 99999, &mut s);
    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 100000, &mut s);
    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 100001, &mut s);
    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 1000000000000000, &mut s);
    digit_compat_low_u64(chinese_number_table, ChineseNumberCase::Lower, 1001000000000101, &mut s);

    assert_eq!("零一十九萬九千九百九十九一億一億零一一極一極零一澗零一百零一", s);
}

#[test]
fn test_digit_compat_ten_thousand_u32() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_compat_ten_thousand_u32(chinese_number_table, ChineseNumberCase::Lower, 0, &mut s);
    digit_compat_ten_thousand_u32(chinese_number_table, ChineseNumberCase::Lower, 1, &mut s);
    digit_compat_ten_thousand_u32(chinese_number_table, ChineseNumberCase::Lower, 10, &mut s);
    digit_compat_ten_thousand_u32(chinese_number_table, ChineseNumberCase::Lower, 99999, &mut s);
    digit_compat_ten_thousand_u32(chinese_number_table, ChineseNumberCase::Lower, 100000, &mut s);
    digit_compat_ten_thousand_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1000000001,
        &mut s,
    );
    digit_compat_ten_thousand_u32(
        chinese_number_table,
        ChineseNumberCase::Lower,
        u32::max_value(),
        &mut s,
    );

    assert_eq!("零一十九萬九千九百九十九十萬十億零一四十二億九千四百九十六萬七千二百九十五", s);
}

#[test]
fn test_digit_compat_ten_thousand_u64() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_compat_ten_thousand_u64(chinese_number_table, ChineseNumberCase::Lower, 0, &mut s);
    digit_compat_ten_thousand_u64(chinese_number_table, ChineseNumberCase::Lower, 1, &mut s);
    digit_compat_ten_thousand_u64(chinese_number_table, ChineseNumberCase::Lower, 10, &mut s);
    digit_compat_ten_thousand_u64(chinese_number_table, ChineseNumberCase::Lower, 99999, &mut s);
    digit_compat_ten_thousand_u64(chinese_number_table, ChineseNumberCase::Lower, 100000, &mut s);
    digit_compat_ten_thousand_u64(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1000000001,
        &mut s,
    );
    digit_compat_ten_thousand_u64(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1000000000000000,
        &mut s,
    );
    digit_compat_ten_thousand_u64(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1001000000000101,
        &mut s,
    );
    digit_compat_ten_thousand_u64(
        chinese_number_table,
        ChineseNumberCase::Lower,
        u64::max_value(),
        &mut s,
    );

    assert_eq!("零一十九萬九千九百九十九十萬十億零一一千兆一千零一兆零一百零一一千八百四十四京六千七百四十四兆零七百三十七億零九百五十五萬一千六百一十五", s);
}

#[test]
fn test_digit_compat_ten_thousand_u128() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_compat_ten_thousand_u128(chinese_number_table, ChineseNumberCase::Lower, 0, &mut s);
    digit_compat_ten_thousand_u128(chinese_number_table, ChineseNumberCase::Lower, 1, &mut s);
    digit_compat_ten_thousand_u128(chinese_number_table, ChineseNumberCase::Lower, 10, &mut s);
    digit_compat_ten_thousand_u128(chinese_number_table, ChineseNumberCase::Lower, 99999, &mut s);
    digit_compat_ten_thousand_u128(chinese_number_table, ChineseNumberCase::Lower, 100000, &mut s);
    digit_compat_ten_thousand_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1000000001,
        &mut s,
    );
    digit_compat_ten_thousand_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1000000000000000,
        &mut s,
    );
    digit_compat_ten_thousand_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1001000000000101,
        &mut s,
    );
    digit_compat_ten_thousand_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        u128::from(u64::max_value()),
        &mut s,
    );
    digit_compat_ten_thousand_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        u128::max_value(),
        &mut s,
    );

    assert_eq!("零一十九萬九千九百九十九十萬十億零一一千兆一千零一兆零一百零一一千八百四十四京六千七百四十四兆零七百三十七億零九百五十五萬一千六百一十五三百四十澗二千八百二十三溝六千六百九十二穰零九百三十八秭四千六百三十四垓六千三百三十七京四千六百零七兆四千三百一十七億六千八百二十一萬一千四百五十五", s);
}

#[test]
fn test_digit_compat_middle_u64() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_compat_middle_u64(chinese_number_table, ChineseNumberCase::Lower, 0, &mut s);
    digit_compat_middle_u64(chinese_number_table, ChineseNumberCase::Lower, 1, &mut s);
    digit_compat_middle_u64(chinese_number_table, ChineseNumberCase::Lower, 10, &mut s);
    digit_compat_middle_u64(chinese_number_table, ChineseNumberCase::Lower, 99999, &mut s);
    digit_compat_middle_u64(chinese_number_table, ChineseNumberCase::Lower, 100000, &mut s);
    digit_compat_middle_u64(chinese_number_table, ChineseNumberCase::Lower, 1000000001, &mut s);
    digit_compat_middle_u64(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1000000000000000,
        &mut s,
    );
    digit_compat_middle_u64(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1001000000000101,
        &mut s,
    );
    digit_compat_middle_u64(
        chinese_number_table,
        ChineseNumberCase::Lower,
        u64::max_value(),
        &mut s,
    );

    assert_eq!("零一十九萬九千九百九十九十萬十億零一一千萬億一千零一萬億零一百零一一千八百四十四兆六千七百四十四萬零七百三十七億零九百五十五萬一千六百一十五", s);
}

#[test]
fn test_digit_compat_middle_u128() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    digit_compat_middle_u128(chinese_number_table, ChineseNumberCase::Lower, 0, &mut s);
    digit_compat_middle_u128(chinese_number_table, ChineseNumberCase::Lower, 1, &mut s);
    digit_compat_middle_u128(chinese_number_table, ChineseNumberCase::Lower, 10, &mut s);
    digit_compat_middle_u128(chinese_number_table, ChineseNumberCase::Lower, 99999, &mut s);
    digit_compat_middle_u128(chinese_number_table, ChineseNumberCase::Lower, 100000, &mut s);
    digit_compat_middle_u128(chinese_number_table, ChineseNumberCase::Lower, 1000000001, &mut s);
    digit_compat_middle_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1000000000000000,
        &mut s,
    );
    digit_compat_middle_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        1001000000000101,
        &mut s,
    );
    digit_compat_middle_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        u128::from(u64::max_value()),
        &mut s,
    );
    digit_compat_middle_u128(
        chinese_number_table,
        ChineseNumberCase::Lower,
        u128::max_value(),
        &mut s,
    );

    assert_eq!("零一十九萬九千九百九十九十萬十億零一一千萬億一千零一萬億零一百零一一千八百四十四兆六千七百四十四萬零七百三十七億零九百五十五萬一千六百一十五三百四十萬二千八百二十三垓六千六百九十二萬零九百三十八京四千六百三十四萬六千三百三十七兆四千六百零七萬四千三百一十七億六千八百二十一萬一千四百五十五", s);
}

#[test]
fn test_fraction_compat_low() {
    let mut s = String::new();

    let chinese_number_table =
        get_chinese_number_table(ChineseVariant::Traditional, ChineseNumberCase::Lower);

    fraction_compat_low(chinese_number_table, ChineseNumberCase::Lower, 0f64, &mut s);
    fraction_compat_low(chinese_number_table, ChineseNumberCase::Lower, 0.01f64, &mut s);
    fraction_compat_low(chinese_number_table, ChineseNumberCase::Lower, 0.1f64, &mut s);
    fraction_compat_low(chinese_number_table, ChineseNumberCase::Lower, 0.55f64, &mut s);

    assert_eq!("零一分一角五角五分", s);
}

#[test]
fn test_chinese_digit_1() {
    assert_eq!(0, chinese_digit_1('零').unwrap());
    assert_eq!(1, chinese_digit_1('壹').unwrap());
    assert_eq!(2, chinese_digit_1('贰').unwrap());
    assert_eq!(3, chinese_digit_1('參').unwrap());
    assert_eq!(4, chinese_digit_1('四').unwrap());
    assert_eq!(5, chinese_digit_1('五').unwrap());
    assert_eq!(6, chinese_digit_1('陸').unwrap());
    assert_eq!(7, chinese_digit_1('七').unwrap());
    assert_eq!(8, chinese_digit_1('八').unwrap());
    assert_eq!(9, chinese_digit_1('玖').unwrap());
}

#[test]
fn test_chinese_digit_10() {
    assert_eq!(10, chinese_digit_10('十', None, None).unwrap());
    assert_eq!(10, chinese_digit_10('壹', Some('十'), None).unwrap());
    assert_eq!(25, chinese_digit_10('贰', Some('拾'), Some('五')).unwrap());
}

#[test]
fn test_chinese_digit_10_compat() {
    assert_eq!(0, chinese_digit_10_compat('零', None, None).unwrap());
    assert_eq!(1, chinese_digit_10_compat('壹', None, None).unwrap());
    assert_eq!(2, chinese_digit_10_compat('贰', None, None).unwrap());
    assert_eq!(10, chinese_digit_10_compat('十', None, None).unwrap());
    assert_eq!(10, chinese_digit_10_compat('壹', Some('十'), None).unwrap());
    assert_eq!(25, chinese_digit_10_compat('贰', Some('拾'), Some('五')).unwrap());
}

#[test]
fn test_chinese_digit_100() {
    assert_eq!(100, chinese_digit_100('壹', '百', None, None, None).unwrap());
    assert_eq!(204, chinese_digit_100('二', '百', Some('零'), Some('四'), None).unwrap());
    assert_eq!(380, chinese_digit_100('三', '百', Some('八'), Some('十'), None).unwrap());
    assert_eq!(999, chinese_digit_100('九', '佰', Some('玖'), Some('拾'), Some('玖')).unwrap());
}

#[test]
fn test_chinese_digit_100_compat() {
    assert_eq!(0, chinese_digit_100_compat('零', None, None, None, None).unwrap());
    assert_eq!(1, chinese_digit_100_compat('壹', None, None, None, None).unwrap());
    assert_eq!(10, chinese_digit_100_compat('十', None, None, None, None).unwrap());
    assert_eq!(10, chinese_digit_100_compat('壹', Some('十'), None, None, None).unwrap());
    assert_eq!(25, chinese_digit_100_compat('贰', Some('拾'), Some('五'), None, None).unwrap());
    assert_eq!(100, chinese_digit_100_compat('壹', Some('百'), None, None, None).unwrap());
    assert_eq!(
        204,
        chinese_digit_100_compat('二', Some('百'), Some('零'), Some('四'), None).unwrap()
    );
    assert_eq!(
        380,
        chinese_digit_100_compat('三', Some('百'), Some('八'), Some('十'), None).unwrap()
    );
    assert_eq!(
        999,
        chinese_digit_100_compat('九', Some('佰'), Some('玖'), Some('拾'), Some('玖')).unwrap()
    );
}

#[test]
fn test_chinese_digit_1000() {
    assert_eq!(1000, chinese_digit_1000('壹', '千', None, None, None, None, None).unwrap());
    assert_eq!(
        1001,
        chinese_digit_1000('一', '仟', Some('零'), Some('壹'), None, None, None).unwrap()
    );
    assert_eq!(
        1010,
        chinese_digit_1000('一', '仟', Some('零'), Some('壹'), Some('拾'), None, None).unwrap()
    );
    assert_eq!(
        1011,
        chinese_digit_1000('一', '仟', Some('零'), Some('壹'), Some('拾'), Some('一'), None)
            .unwrap()
    );
    assert_eq!(
        2300,
        chinese_digit_1000('二', '千', Some('三'), Some('百'), None, None, None).unwrap()
    );
    assert_eq!(
        2301,
        chinese_digit_1000('二', '千', Some('三'), Some('百'), Some('零'), Some('一'), None)
            .unwrap()
    );
    assert_eq!(
        2320,
        chinese_digit_1000('二', '千', Some('三'), Some('百'), Some('二'), Some('十'), None)
            .unwrap()
    );
    assert_eq!(
        9999,
        chinese_digit_1000('九', '千', Some('玖'), Some('佰'), Some('玖'), Some('十'), Some('玖'))
            .unwrap()
    );
}

#[test]
fn test_chinese_digit_1000_compat() {
    assert_eq!(0, chinese_digit_1000_compat('零', None, None, None, None, None, None).unwrap());
    assert_eq!(1, chinese_digit_1000_compat('壹', None, None, None, None, None, None).unwrap());
    assert_eq!(10, chinese_digit_1000_compat('十', None, None, None, None, None, None).unwrap());
    assert_eq!(
        10,
        chinese_digit_1000_compat('壹', Some('十'), None, None, None, None, None).unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_1000_compat('贰', Some('拾'), Some('五'), None, None, None, None).unwrap()
    );
    assert_eq!(
        100,
        chinese_digit_1000_compat('壹', Some('百'), None, None, None, None, None).unwrap()
    );
    assert_eq!(
        204,
        chinese_digit_1000_compat('二', Some('百'), Some('零'), Some('四'), None, None, None)
            .unwrap()
    );
    assert_eq!(
        380,
        chinese_digit_1000_compat('三', Some('百'), Some('八'), Some('十'), None, None, None)
            .unwrap()
    );
    assert_eq!(
        999,
        chinese_digit_1000_compat('九', Some('佰'), Some('玖'), Some('拾'), Some('玖'), None, None)
            .unwrap()
    );
    assert_eq!(
        1000,
        chinese_digit_1000_compat('壹', Some('千'), None, None, None, None, None).unwrap()
    );
    assert_eq!(
        1001,
        chinese_digit_1000_compat('一', Some('仟'), Some('零'), Some('壹'), None, None, None)
            .unwrap()
    );
    assert_eq!(
        1010,
        chinese_digit_1000_compat('一', Some('仟'), Some('零'), Some('壹'), Some('拾'), None, None)
            .unwrap()
    );
    assert_eq!(
        1011,
        chinese_digit_1000_compat(
            '一',
            Some('仟'),
            Some('零'),
            Some('壹'),
            Some('拾'),
            Some('一'),
            None
        )
        .unwrap()
    );
    assert_eq!(
        2300,
        chinese_digit_1000_compat('二', Some('千'), Some('三'), Some('百'), None, None, None)
            .unwrap()
    );
    assert_eq!(
        2301,
        chinese_digit_1000_compat(
            '二',
            Some('千'),
            Some('三'),
            Some('百'),
            Some('零'),
            Some('一'),
            None
        )
        .unwrap()
    );
    assert_eq!(
        2320,
        chinese_digit_1000_compat(
            '二',
            Some('千'),
            Some('三'),
            Some('百'),
            Some('二'),
            Some('十'),
            None
        )
        .unwrap()
    );
    assert_eq!(
        9999,
        chinese_digit_1000_compat(
            '九',
            Some('千'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('玖')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_10_000_low_compat() {
    assert_eq!(
        0,
        chinese_digit_10_000_low_compat('零', None, None, None, None, None, None, None, None)
            .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_10_000_low_compat('壹', None, None, None, None, None, None, None, None)
            .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_10_000_low_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_10_000_low_compat('一', Some('萬'), None, None, None, None, None, None, None)
            .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_10_000_low_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        11000,
        chinese_digit_10_000_low_compat(
            '一',
            Some('萬'),
            Some('一'),
            Some('仟'),
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        99999,
        chinese_digit_10_000_low_compat(
            '九',
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_100_000_low_compat() {
    assert_eq!(
        0,
        chinese_digit_100_000_low_compat(
            '零', None, None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_100_000_low_compat(
            '壹', None, None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_100_000_low_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_100_000_low_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_100_000_low_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_100_000_low_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_100_000_low_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110000,
        chinese_digit_100_000_low_compat(
            '一',
            Some('億'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110001,
        chinese_digit_100_000_low_compat(
            '一',
            Some('億'),
            Some('一'),
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        999999,
        chinese_digit_100_000_low_compat(
            '九',
            Some('億'),
            Some('九'),
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_1_000_000_low_compat() {
    assert_eq!(
        0,
        chinese_digit_1_000_000_low_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_1_000_000_low_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_1_000_000_low_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000001,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('兆'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1010000,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('兆'),
            Some('零'),
            Some('壹'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1010001,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('兆'),
            Some('零'),
            Some('壹'),
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1100000,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('兆'),
            Some('一'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1100001,
        chinese_digit_1_000_000_low_compat(
            '一',
            Some('兆'),
            Some('一'),
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        9999999,
        chinese_digit_1_000_000_low_compat(
            '九',
            Some('兆'),
            Some('玖'),
            Some('億'),
            Some('九'),
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_10_000_000_low_compat() {
    assert_eq!(
        0,
        chinese_digit_10_000_000_low_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_10_000_000_low_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_10_000_000_low_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000001,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('兆'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('京'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000001,
        chinese_digit_10_000_000_low_compat(
            '一',
            Some('京'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        99999999,
        chinese_digit_10_000_000_low_compat(
            '九',
            Some('京'),
            Some('九'),
            Some('兆'),
            Some('玖'),
            Some('億'),
            Some('九'),
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_100_000_000_low_compat() {
    assert_eq!(
        0,
        chinese_digit_100_000_000_low_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_100_000_000_low_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_100_000_000_low_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000001,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('兆'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('京'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000001,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('京'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('垓'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000001,
        chinese_digit_100_000_000_low_compat(
            '一',
            Some('垓'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        999999999,
        chinese_digit_100_000_000_low_compat(
            '九',
            Some('垓'),
            Some('九'),
            Some('京'),
            Some('九'),
            Some('兆'),
            Some('玖'),
            Some('億'),
            Some('九'),
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_1_000_000_000_low_compat() {
    assert_eq!(
        0,
        chinese_digit_1_000_000_000_low_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_1_000_000_000_low_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_1_000_000_000_low_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000001,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('兆'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('京'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000001,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('京'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('垓'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000001,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('垓'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('秭'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000001,
        chinese_digit_1_000_000_000_low_compat(
            '一',
            Some('秭'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        9999999999,
        chinese_digit_1_000_000_000_low_compat(
            '九',
            Some('秭'),
            Some('九'),
            Some('垓'),
            Some('九'),
            Some('京'),
            Some('九'),
            Some('兆'),
            Some('玖'),
            Some('億'),
            Some('九'),
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_1_000_000_000_000_000_low_compat() {
    assert_eq!(
        0,
        chinese_digit_1_000_000_000_000_000_low_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_1_000_000_000_000_000_low_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000000,
        chinese_digit_1_000_000_000_000_000_low_compat(
            '一',
            Some('極'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000001,
        chinese_digit_1_000_000_000_000_000_low_compat(
            '一',
            Some('極'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        9999999999999999,
        chinese_digit_1_000_000_000_000_000_low_compat(
            '九',
            Some('極'),
            Some('九'),
            Some('載'),
            Some('九'),
            Some('正'),
            Some('九'),
            Some('澗'),
            Some('玖'),
            Some('溝'),
            Some('九'),
            Some('穰'),
            Some('九'),
            Some('秭'),
            Some('玖'),
            Some('垓'),
            Some('玖'),
            Some('京'),
            Some('九'),
            Some('兆'),
            Some('玖'),
            Some('億'),
            Some('九'),
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_10_000_ten_thousand_compat() {
    assert_eq!(
        0,
        chinese_digit_10_000_ten_thousand_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_10_000_ten_thousand_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_10_000_ten_thousand_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        11000,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('一'),
            Some('仟'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        99999,
        chinese_digit_10_000_ten_thousand_compat(
            '九',
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九'),
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_10_000_ten_thousand_compat(
            '十',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('十'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_10_000_ten_thousand_compat(
            '十',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110000,
        chinese_digit_10_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110001,
        chinese_digit_10_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1010000,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10010000,
        chinese_digit_10_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        99999999,
        chinese_digit_10_000_ten_thousand_compat(
            '九',
            Some('千'),
            Some('九'),
            Some('百'),
            Some('玖'),
            Some('拾'),
            Some('九'),
            Some('萬'),
            Some('玖'),
            Some('仟'),
            Some('玖'),
            Some('百'),
            Some('九'),
            Some('十'),
            Some('九')
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_100_000_000_ten_thousand_compat() {
    assert_eq!(
        0,
        chinese_digit_100_000_000_ten_thousand_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_100_000_000_ten_thousand_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_100_000_000_ten_thousand_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        11000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('一'),
            Some('仟'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        99999,
        chinese_digit_100_000_000_ten_thousand_compat(
            '九',
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '十',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('十'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_100_000_000_ten_thousand_compat(
            '十',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110001,
        chinese_digit_100_000_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1010000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10010000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000001,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100010001,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('一'),
            Some('萬'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '十',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000001,
        chinese_digit_100_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('億'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_1_000_000_000_000_ten_thousand_compat() {
    assert_eq!(
        0,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        11000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('一'),
            Some('仟'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        99999,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '九',
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '十',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('十'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '十',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110001,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1010000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10010000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000001,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100010001,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('一'),
            Some('萬'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '十',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000001,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('億'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '十',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000000,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000001,
        chinese_digit_1_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('兆'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_10_000_000_000_000_000_ten_thousand_compat() {
    assert_eq!(
        0,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        11000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('萬'),
            Some('一'),
            Some('仟'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        99999,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '九',
            Some('萬'),
            Some('九'),
            Some('仟'),
            Some('玖'),
            Some('佰'),
            Some('玖'),
            Some('十'),
            Some('九'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '十',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('十'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '十',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        110001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '十',
            Some('壹'),
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1010000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10010000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('零'),
            Some('一'),
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100010001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('一'),
            Some('萬'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '十',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('億'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '十',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('兆'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('京'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '十',
            Some('京'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('百'),
            Some('京'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000000000000,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('京'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000000000001,
        chinese_digit_10_000_000_000_000_000_ten_thousand_compat(
            '一',
            Some('千'),
            Some('京'),
            Some('零'),
            Some('一'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_100_000_000_middle_compat() {
    assert_eq!(
        0,
        chinese_digit_100_000_000_middle_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_100_000_000_middle_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_100_000_000_middle_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_100_000_000_middle_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_100_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000,
        chinese_digit_100_000_000_middle_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000001,
        chinese_digit_100_000_000_middle_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000,
        chinese_digit_100_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000001,
        chinese_digit_100_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
}

#[test]
fn test_chinese_digit_10_000_000_000_000_000_middle_compat() {
    assert_eq!(
        0,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '零', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        1,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '壹', None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None, None, None, None, None, None, None, None, None, None,
            None, None, None, None, None
        )
        .unwrap()
    );
    assert_eq!(
        25,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '贰',
            Some('拾'),
            Some('五'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('萬'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10001,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000001,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000000,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('億'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        1000000000001,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('億'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000000000,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        10000000000000001,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('兆'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000000000000,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('兆'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
    assert_eq!(
        100000000000000000001,
        chinese_digit_10_000_000_000_000_000_middle_compat(
            '一',
            Some('萬'),
            Some('兆'),
            Some('零'),
            Some('壹'),
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None,
            None
        )
        .unwrap()
    );
}
