extern crate proc_macro;

use proc_macro::TokenStream;
use quote::quote;
use syn::parse::{Parse, ParseStream};
use syn::{parenthesized, parse, parse2, DeriveInput, Generics, Ident, Result};

struct ContextType {
    name: Ident,
    generics: Generics,
}

impl Parse for ContextType {
    fn parse(input: ParseStream) -> Result<Self> {
        let content;
        parenthesized!(content in input);

        Ok(ContextType {
            name: content.parse()?,
            generics: content.parse()?,
        })
    }
}

#[proc_macro_derive(Server, attributes(context_type))]
pub fn checkout_server_derive(input: TokenStream) -> TokenStream {
    let ast = parse(input).unwrap();

    impl_checkout_server(&ast)
}

fn impl_checkout_server(ast: &DeriveInput) -> TokenStream {
    let server_type_name = &ast.ident;

    let context_type_attribute = ast
        .attrs
        .iter()
        .filter(|a| a.path.segments.len() == 1 && a.path.segments[0].ident == "context_type")
        .nth(0)
        .expect("The context_type attribute required to derive Server. This should be a type that implements Context.");

    let context_type: ContextType =
        parse2(context_type_attribute.tokens.clone()).expect("invalid context_type");

    let context_type_name = &context_type.name;

    let mut impl_generics_container = context_type.generics.clone();
    impl_generics_container.params.clear();
    impl_generics_container
        .params
        .push(syn::parse_quote!('common));

    let mut ty_generics_container = context_type.generics.clone();
    ty_generics_container.params.clear();

    for _ in context_type.generics.lifetimes() {
        ty_generics_container
            .params
            .push(syn::parse_quote!('common));
    }

    let (impl_generics, _, _) = impl_generics_container.split_for_impl();
    let (_, ty_generics, _) = ty_generics_container.split_for_impl();
    let common_lifetime_def = syn::LifetimeDef::new(syn::parse_quote!('common));
    let common_lifetime = common_lifetime_def.lifetime;

    let gen = quote! {
        mod checkout_server {
            pub use std::sync::Arc;
            pub use checkout_core::cloud_pubsub;
            pub use checkout_core::dropshot::ApiDescription;
            pub use checkout_core::dropshot::ConfigDropshot;
            pub use checkout_core::dropshot::ConfigLogging;
            pub use checkout_core::dropshot::ConfigLoggingLevel;
            pub use checkout_core::dropshot::HttpServerStarter;
            pub use checkout_core::error::Error;
            pub use checkout_core::error::new_server_error;
            pub use checkout_core::server::Server;
            pub use checkout_core::internal_context::InternalContext;

            use std::env;
            use checkout_core::dropshot;
            use checkout_core::dropshot::endpoint;
            use checkout_core::dropshot::RequestContext;
            use checkout_core::dropshot::Path;
            use checkout_core::dropshot::TypedBody;
            use checkout_core::dropshot::HttpError;
            use checkout_core::dropshot::HttpResponseOk;
            use checkout_core::http::StatusCode;
            use checkout_core::checkout::Checkout;
            use checkout_core::payment::Payment;
            use checkout_core::payment::PaymentProcessor;
            use checkout_core::payment::handlers::process_handler;
            use checkout_core::customer::Contact;
            use checkout_core::customer::Address;
            use checkout_core::shipping::FulfillmentTypeSelection;
            use checkout_core::server::IdParam;
            use checkout_core::server::CreateParams;
            use checkout_core::server::AddOrRemoveItemParams;
            use checkout_core::transaction::TransactionController;
            use checkout_core::context::Context;
            use checkout_core::context::ContextProvider;
            use checkout_core::checkout::handlers::{create_handler, add_item_handler, remove_item_handler, confirm_items_handler, update_contact_handler, update_shipping_address_handler, update_fulfillment_type_handler, initiate_payment_handler};

            pub struct Config {
                pub google_application_credentials: String,
                pub payment_processing_topic: String,
                pub checkout_completion_topic: String,
                pub bind_address: String,
            }

            pub fn get_config() -> Config {
                Config {
                    google_application_credentials: env::var("GOOGLE_APPLICATION_CREDENTIALS").unwrap(),
                    payment_processing_topic: env::var("PAYMENT_PROCESSING_TOPIC").unwrap(),
                    checkout_completion_topic: env::var("CHECKOUT_COMPLETION_TOPIC").unwrap(),
                    bind_address: env::var("BIND_ADDRESS").unwrap(),
                }
            }

            pub struct ServerContext {
                pub internal_context: InternalContext,
                pub context_provider: Arc<super::#server_type_name>,
            }

            #[derive(Debug)]
            struct Message(String);

            impl cloud_pubsub::FromPubSubMessage for Message {
                fn from(message: cloud_pubsub::EncodedMessage) -> Result<Self, cloud_pubsub::error::Error> {
                    match message.decode() {
                        Ok(bytes) => Ok(Message(String::from_utf8_lossy(&bytes).into_owned())),
                        Err(e) => Err(cloud_pubsub::error::Error::from(e)),
                    }
                }
            }

            pub async fn start_payment_processing(
                client: Arc<cloud_pubsub::Client>,
                subscription_name: String,
                internal_ctx: InternalContext,
                context_provider: Arc<super::#server_type_name>,
            ) {
                let subscription = client.subscribe(subscription_name);

                while subscription.client().is_running() {
                    match subscription.get_messages::<Message>().await {
                        Ok(messages) => {
                            for (result, ack_id) in messages {
                                match result {
                                    Ok(message) => {
                                        println!("Received: {:?}", message);
                                        match context_provider.new_context().await {
                                            Ok(mut ctx) => {
                                                let processing_result = process_handler(&internal_ctx, &mut ctx, message.0).await;
                                                if processing_result.is_ok() {
                                                    subscription.acknowledge_messages(vec![ack_id]).await;
                                                }
                                            }
                                            Err(_) => {}
                                        }
                                    }
                                    Err(e) => eprintln!("Failed to parse message: {}", e),
                                }
                            }
                        }
                        Err(e) => eprintln!("Failed to pull PubSub messages: {}", e),
                    }
                }
            }

            #[endpoint {
                method = POST,
                path = "/checkout",
            }]
            pub async fn create_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                body: TypedBody<CreateParams>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let params = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match create_handler(&mut ctx, params.currency.clone(), params.contact.clone(), params.shipping_address.clone()).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/add_item",
            }]
            pub async fn add_item_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<AddOrRemoveItemParams>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let path_params = path.into_inner();
                let body_params = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match add_item_handler(&mut ctx, &path_params.id, body_params.sku.clone(), body_params.quantity).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/remove_item",
            }]
            pub async fn remove_item_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<AddOrRemoveItemParams>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let path_params = path.into_inner();
                let body_params = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match remove_item_handler(&mut ctx, &path_params.id, body_params.sku.clone(), body_params.quantity).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/confirm_items",
            }]
            pub async fn confirm_items_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let path_params = path.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();

                match confirm_items_handler(&mut ctx, &path_params.id).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/update_contact",
            }]
            pub async fn update_contact_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<Contact>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let path_params = path.into_inner();
                let contact = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match update_contact_handler(&mut ctx, &path_params.id, contact).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/update_shipping_address",
            }]
            pub async fn update_shipping_address_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<Address>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let path_params = path.into_inner();
                let shipping_address = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match update_shipping_address_handler(&mut ctx, &path_params.id, shipping_address).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/update_fulfillment_type",
            }]
            pub async fn update_fulfillment_type_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<FulfillmentTypeSelection>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let path_params = path.into_inner();
                let fulfillment_type = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match update_fulfillment_type_handler(&mut ctx, &path_params.id, fulfillment_type).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/initiate_payment",
            }]
            pub async fn initiate_payment_endpoint(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<<super::#context_type_name<'_> as PaymentProcessor>::InitiateArgs>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let path_params = path.into_inner();
                let args = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match initiate_payment_handler(&mut ctx, &path_params.id, args).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }
        }

        #[async_trait]
        impl #impl_generics checkout_server::Server<#common_lifetime, #context_type_name #ty_generics> for #server_type_name {
            type Context = checkout_server::ServerContext;

            fn api_description() -> checkout_server::ApiDescription<checkout_server::ServerContext> {
                let mut api = checkout_server::ApiDescription::<checkout_server::ServerContext>::new();
                api.register(checkout_server::create_endpoint).unwrap();
                api.register(checkout_server::add_item_endpoint).unwrap();
                api.register(checkout_server::remove_item_endpoint).unwrap();
                api.register(checkout_server::confirm_items_endpoint).unwrap();
                api.register(checkout_server::update_contact_endpoint).unwrap();
                api.register(checkout_server::update_shipping_address_endpoint).unwrap();
                api.register(checkout_server::update_fulfillment_type_endpoint).unwrap();
                api.register(checkout_server::initiate_payment_endpoint).unwrap();

                api
            }

            async fn start(
                self
            ) -> Result<(), checkout_server::Error> {
                let log = checkout_server::ConfigLogging::StderrTerminal {
                    level: checkout_server::ConfigLoggingLevel::Info,
                }
                .to_logger("minimal-example")
                .map_err(|_| checkout_server::new_server_error("failed to setup logger"))?;

                let api = Self::api_description();

                let config = checkout_server::get_config();

                let pubsub_client = checkout_server::Arc::new(checkout_server::cloud_pubsub::Client::new(config.google_application_credentials.clone()).await?);
                let payment_processing_topic = checkout_server::Arc::new(pubsub_client.topic(config.payment_processing_topic.clone()));
                let checkout_completion_topic: checkout_server::Arc::new(pubsub_client.topic(config.checkout_completion_topic.clone()));
                let internal_context = checkout_server::InternalContext::new(payment_processing_topic, checkout_completion_topic).await.map_err(|_| checkout_server::new_server_error("failed to create internal context"))?;

                let context_provider = checkout_server::Arc::new(self);

                checkout_server::HttpServerStarter::new(
                    &checkout_server::ConfigDropshot {
                        bind_address: config.bind_address.parse().unwrap(),
                        request_body_max_bytes: 2048 * 1024,
                    },
                    api,
                    checkout_server::ServerContext {
                        internal_context,
                        context_provider: context_provider.clone(),
                    },
                    &log,
                )
                .map_err(|_| checkout_server::new_server_error("failed to set up api server"))?
                .start()
                .await
                .map_err(|_| checkout_server::new_server_error("failed to start server"))
            }
        }
    };
    gen.into()
}
