extern crate proc_macro;

use proc_macro::TokenStream;
use quote::quote;
use syn::parse::{Parse, ParseStream};
use syn::{parenthesized, parse, parse2, DeriveInput, Ident, Result};

struct ServerStruct {
    name: Ident,
}

impl Parse for ServerStruct {
    fn parse(input: ParseStream) -> Result<Self> {
        let content;
        parenthesized!(content in input);

        Ok(ServerStruct {
            name: content.parse()?,
        })
    }
}

#[proc_macro_derive(Server, attributes(struct_name))]
pub fn checkout_server_derive(input: TokenStream) -> TokenStream {
    let ast = parse(input).unwrap();

    // Build the trait implementation
    impl_checkout_server(&ast)
}

fn impl_checkout_server(ast: &DeriveInput) -> TokenStream {
    // panic if there are type generics defined on the Context type
    for _ in ast.generics.type_params() {
        panic!("can't have generic types defined on Context type")
    }

    let context_struct_name = &ast.ident;

    let server_struct_attribute = ast
        .attrs
        .iter()
        .filter(|a| a.path.segments.len() == 1 && a.path.segments[0].ident == "struct_name")
        .nth(0)
        .expect("struct_name attribute required to derive Server. This should be the name of the struct where ContextProvider is implemented");

    let server_struct: ServerStruct =
        parse2(server_struct_attribute.tokens.clone()).expect("invalid struct_name");
    let server_struct_name = &server_struct.name;

    let mut impl_generics_container = ast.generics.clone();
    impl_generics_container.params.clear();
    impl_generics_container
        .params
        .push(syn::parse_quote!('common));

    let mut ty_generics_container = ast.generics.clone();
    ty_generics_container.params.clear();

    for _ in ast.generics.lifetimes() {
        ty_generics_container
            .params
            .push(syn::parse_quote!('common));
    }

    let (impl_generics, _, _) = impl_generics_container.split_for_impl();
    let (_, ty_generics, _) = ty_generics_container.split_for_impl();
    let common_lifetime_def = syn::LifetimeDef::new(syn::parse_quote!('common));
    let common_lifetime = common_lifetime_def.lifetime;

    let gen = quote! {
        mod checkout_server {
            pub use checkout_core::dropshot::ApiDescription;
            pub use checkout_core::dropshot::ConfigDropshot;
            pub use checkout_core::dropshot::ConfigLogging;
            pub use checkout_core::dropshot::ConfigLoggingLevel;
            pub use checkout_core::dropshot::HttpServerStarter;
            pub use checkout_core::error::Error;
            pub use checkout_core::server::Server;

            use std::sync::Arc;
            use checkout_core::dropshot;
            use checkout_core::dropshot::endpoint;
            use checkout_core::dropshot::RequestContext;
            use checkout_core::dropshot::TypedBody;
            use checkout_core::dropshot::HttpError;
            use checkout_core::dropshot::HttpResponseOk;
            use checkout_core::http::StatusCode;
            use checkout_core::checkout::Checkout;
            use checkout_core::server::CreateRequest;
            use checkout_core::store::CheckoutStore;
            use checkout_core::transaction::TransactionController;
            use checkout_core::context::Context;
            use checkout_core::context::ContextProvider;

            pub struct ServerContext {
                pub context_provider: super::#server_struct_name
            }

            #[endpoint {
                method = POST,
                path = "/checkout",
            }]
            pub async fn create_handler(
                req_ctx: Arc<RequestContext<ServerContext>>,
                body_params: TypedBody<CreateRequest>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let params = body_params.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match Checkout::create(&mut ctx, params.currency.clone()).await {
                    Ok(co) => {
                        ctx.set_checkout(&co.id, &co).await.unwrap();
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: Some(err.code.clone()),
                            external_message: err.message.clone(),
                            internal_message: err.message.clone(),
                        })
                    }
                }
            }
        }

        impl #impl_generics checkout_server::Server<#common_lifetime, #context_struct_name #ty_generics> for #server_struct_name {
            type Context = checkout_server::ServerContext;

            fn api_description() -> checkout_server::ApiDescription<checkout_server::ServerContext> {
                let mut api = checkout_server::ApiDescription::<checkout_server::ServerContext>::new();
                api.register(checkout_server::create_handler).unwrap();
                api
            }

            fn new_server(
                self,
                bind_address: &str,
            ) -> Result<
                checkout_server::HttpServerStarter<checkout_server::ServerContext>,
                checkout_server::Error,
            > {
                let log = checkout_server::ConfigLogging::StderrTerminal {
                    level: checkout_server::ConfigLoggingLevel::Info,
                }
                .to_logger("minimal-example")
                .map_err(|_| checkout_server::Error {
                    code: "500".to_string(),
                    message: "failed to bind server".to_string(),
                })?;

                let api = Self::api_description();

                checkout_server::HttpServerStarter::new(
                    &checkout_server::ConfigDropshot {
                        bind_address: bind_address.parse().unwrap(),
                        request_body_max_bytes: 2048 * 1024,
                    },
                    api,
                    checkout_server::ServerContext {
                        context_provider: self,
                    },
                    &log,
                )
                .map_err(|_| checkout_server::Error {
                    code: "500".to_string(),
                    message: "failed to bind server".to_string(),
                })
            }
        }
    };
    gen.into()
}
