extern crate proc_macro;

use proc_macro::TokenStream;
use quote::quote;
use syn::parse::{Parse, ParseStream};
use syn::{parse, parse2, DeriveInput, Ident, Result};

struct ServerStruct {
    name: Ident,
}

impl Parse for ServerStruct {
    fn parse(input: ParseStream) -> Result<Self> {
        Ok(ServerStruct {
            name: input.parse()?,
        })
    }
}

#[proc_macro_derive(Server, attributes(struct_name))]
pub fn checkout_server_derive(input: TokenStream) -> TokenStream {
    let ast = parse(input).unwrap();

    // Build the trait implementation
    impl_checkout_server(&ast)
}

fn impl_checkout_server(ast: &DeriveInput) -> TokenStream {
    let context_struct_name = &ast.ident;

    let server_struct_attribute = ast
        .attrs
        .iter()
        .filter(|a| a.path.segments.len() == 1 && a.path.segments[0].ident == "struct_name")
        .nth(0)
        .expect("struct_name attribute required to derive Server. This should be the name of the struct where ContextProvider is implemented");

    let server_struct: ServerStruct =
        parse2(server_struct_attribute.tokens.clone()).expect("invalid struct_name");

    let server_struct_name = &server_struct.name;

    let gen = quote! {
        mod checkout_server {
            pub use checkout_core::dropshot::endpoint;
            pub use checkout_core::dropshot::ApiDescription;
            pub use checkout_core::dropshot::ConfigDropshot;
            pub use checkout_core::dropshot::ConfigLogging;
            pub use checkout_core::dropshot::ConfigLoggingLevel;
            pub use checkout_core::dropshot::HttpError;
            pub use checkout_core::dropshot::HttpResponseOk;
            pub use checkout_core::dropshot::HttpServerStarter;
            pub use checkout_core::dropshot::RequestContext;
            pub use checkout_core::dropshot::TypedBody;
            pub use checkout_core::http::StatusCode;
            pub use checkout_core::{
                error::Error, money::Currency, Checkout, Context, ContextProvider, server::Server,
            };

            use std::sync::Arc;
            use checkout_core::dropshot;
            use checkout_core::server::CreateRequest;

            pub struct ServerContext {
                pub context_provider: super::#server_struct_name
            }

            #[endpoint {
                method = POST,
                path = "/create",
            }]
            pub async fn create_handler(
                req_ctx: Arc<RequestContext<ServerContext>>,
                body_params: TypedBody<CreateRequest>,
            ) -> Result<HttpResponseOk<Checkout>, HttpError> {
                let params = body_params.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match Checkout::create(&mut ctx, params.currency.clone()).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(_) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(HttpError {
                            status_code: StatusCode::BAD_REQUEST,
                            error_code: None,
                            external_message: "unavailable".to_string(),
                            internal_message: "unavailable".to_string(),
                        })
                    }
                }
            }
        }

        impl checkout_server::Server<#context_struct_name> for #server_struct_name {
            type Context = checkout_server::ServerContext;

            fn api_description() -> checkout_server::ApiDescription<checkout_server::ServerContext> {
                let mut api = checkout_server::ApiDescription::<checkout_server::ServerContext>::new();
                api.register(checkout_server::create_handler).unwrap();
                api
            }

            fn new_server(
                self,
                bind_address: &str,
            ) -> Result<
                checkout_server::HttpServerStarter<checkout_server::ServerContext>,
                checkout_server::Error,
            > {
                let log = checkout_server::ConfigLogging::StderrTerminal {
                    level: checkout_server::ConfigLoggingLevel::Info,
                }
                .to_logger("minimal-example")
                .map_err(|_| checkout_server::Error {
                    code: "500".to_string(),
                    message: "failed to bind server".to_string(),
                })?;

                let api = Self::api_description();

                checkout_server::HttpServerStarter::new(
                    &checkout_server::ConfigDropshot {
                        bind_address: bind_address.parse().unwrap(),
                        request_body_max_bytes: 2048 * 1024,
                    },
                    api,
                    checkout_server::ServerContext {
                        context_provider: self,
                    },
                    &log,
                )
                .map_err(|_| checkout_server::Error {
                    code: "500".to_string(),
                    message: "failed to bind server".to_string(),
                })
            }
        }
    };
    gen.into()
}
