extern crate proc_macro;

use proc_macro::TokenStream;
use quote::quote;
use syn::parse::{Parse, ParseStream};
use syn::{parenthesized, parse, parse2, DeriveInput, Generics, Ident, Result};

struct ContextType {
    name: Ident,
    generics: Generics,
}

impl Parse for ContextType {
    fn parse(input: ParseStream) -> Result<Self> {
        let content;
        parenthesized!(content in input);

        Ok(ContextType {
            name: content.parse()?,
            generics: content.parse()?,
        })
    }
}

struct PaymentWebhookHandler {
    name: Ident,
}

impl Parse for PaymentWebhookHandler {
    fn parse(input: ParseStream) -> Result<Self> {
        let content;
        parenthesized!(content in input);

        Ok(PaymentWebhookHandler {
            name: content.parse()?,
        })
    }
}

#[proc_macro_derive(Server, attributes(context_type, payment_webhook_handler))]
pub fn checkout_server_derive(input: TokenStream) -> TokenStream {
    let ast = parse(input).unwrap();

    impl_checkout_server(&ast)
}

fn impl_checkout_server(ast: &DeriveInput) -> TokenStream {
    let server_type_name = &ast.ident;

    let context_type_attribute = ast
        .attrs
        .iter()
        .filter(|a| a.path.segments.len() == 1 && a.path.segments[0].ident == "context_type")
        .nth(0)
        .expect("The context_type attribute required to derive Server. This should be a type that implements Context.");

    let context_type: ContextType =
        parse2(context_type_attribute.tokens.clone()).expect("invalid context_type");

    let context_type_name = &context_type.name;

    let mut impl_generics_container = context_type.generics.clone();
    impl_generics_container.params.clear();
    impl_generics_container
        .params
        .push(syn::parse_quote!('common));

    let mut ty_generics_container = context_type.generics.clone();
    ty_generics_container.params.clear();

    for _ in context_type.generics.lifetimes() {
        ty_generics_container
            .params
            .push(syn::parse_quote!('common));
    }

    let mut ty_static_generics_container = context_type.generics.clone();
    ty_static_generics_container.params.clear();

    for _ in context_type.generics.lifetimes() {
        ty_static_generics_container
            .params
            .push(syn::parse_quote!('static));
    }

    let (impl_generics, _, _) = impl_generics_container.split_for_impl();
    let (_, ty_generics, _) = ty_generics_container.split_for_impl();
    let (_, ty_static_generics, _) = ty_static_generics_container.split_for_impl();
    let common_lifetime_def = syn::LifetimeDef::new(syn::parse_quote!('common));
    let common_lifetime = common_lifetime_def.lifetime;

    let maybe_pwh_attr = ast
        .attrs
        .iter()
        .filter(|a| {
            a.path.segments.len() == 1 && a.path.segments[0].ident == "payment_webhook_handler"
        })
        .nth(0);

    let (payment_webhook_endpoint_def, payment_webhook_endpoint_reg) = match maybe_pwh_attr {
        Some(pwh_attr) => {
            let pwh: PaymentWebhookHandler =
                parse2(pwh_attr.tokens.clone()).expect("invalid Payment Webhook Handler");
            let pwh_name = pwh.name;

            let endpoint_definition = quote! {
                #[endpoint {
                    method = POST,
                    path = "/payment_webhook",
                }]
                pub async fn payment_webhook(
                    req_ctx: Arc<RequestContext<ServerContext>>,
                    body: UntypedBody,
                ) -> Result<HttpResponseOk<()>, HttpError> {
                    let internal_ctx = req_ctx
                        .context()
                        .internal_context
                        .clone();
                    let mut ctx = req_ctx
                        .context()
                        .context_provider
                        .new_context()
                        .await
                        .unwrap();
                    let req = req_ctx.request.clone();
                    let guard = req.lock().await;
                    let val = guard.deref();
                    let headers = val.headers().clone();

                    let maybe_callback_args = super::#pwh_name(headers, body).await.map_err(|e| {
                        let out: HttpError = e.into();
                        out
                    })?;

                    if let Some(callback_args) = maybe_callback_args {
                        internal_ctx
                            .payment_callbacks_topic
                            .clone()
                            .publish(callback_args)
                            .await
                            .map_err(|_| {
                                let out: HttpError = new_application_error(
                                    "UPSTREAM_ERROR",
                                    "couldn't send message to payment callbacks topic",
                                ).into();
                                out
                            })?;
                    }

                    Ok(HttpResponseOk(()))
                }
            };

            let endpoint_registration = quote! {
                api.register(checkout_server::payment_webhook).unwrap();
            };

            (endpoint_definition, endpoint_registration)
        }
        None => (quote!(), quote!()),
    };

    let gen = quote! {
        mod checkout_server {
            pub use std::sync::Arc;
            pub use std::time::Duration;
            pub use checkout_core::dropshot::ApiDescription;
            pub use checkout_core::dropshot::ConfigDropshot;
            pub use checkout_core::dropshot::ConfigLogging;
            pub use checkout_core::dropshot::ConfigLoggingLevel;
            pub use checkout_core::dropshot::HttpServerStarter;
            pub use checkout_core::error::Error;
            pub use checkout_core::error::new_server_error;
            pub use checkout_core::server::Server;
            pub use checkout_core::server::spawn_token_renew;
            pub use checkout_core::internal_context::InternalContext;
            pub use checkout_core::db::DBClient;

            use std::env;
            use std::ops::Deref;
            use tokio::task;
            use tokio::time::timeout;
            use checkout_core::cloud_pubsub;
            use checkout_core::schemars::JsonSchema;
            use checkout_core::dropshot;
            use checkout_core::serde::{Serialize, Deserialize};
            use checkout_core::serde_json;
            use checkout_core::dropshot::endpoint;
            use checkout_core::dropshot::RequestContext;
            use checkout_core::dropshot::Path;
            use checkout_core::dropshot::TypedBody;
            use checkout_core::dropshot::UntypedBody;
            use checkout_core::dropshot::HttpError;
            use checkout_core::dropshot::HttpResponseOk;
            use checkout_core::http::StatusCode;
            use checkout_core::checkout::Checkout;
            use checkout_core::payment::{Payment, CallbackArgs};
            use checkout_core::payment::PaymentProcessor;
            use checkout_core::customer::Contact;
            use checkout_core::customer::Address;
            use checkout_core::money::Currency;
            use checkout_core::shipping::FulfillmentSelection;
            use checkout_core::transaction::TransactionController;
            use checkout_core::context::{Context, ClientContext};
            use checkout_core::context::ClientContextProvider;
            use checkout_core::master_context::MasterContext;
            use checkout_core::order::{Order, LookupKey};
            use checkout_core::error::new_application_error;
            use checkout_core::checkout::handlers::{
                create_handler,
                add_item_handler,
                remove_item_handler,
                confirm_items_handler,
                update_contact_handler,
                update_shipping_address_handler,
                update_fulfillment_handler,
                initiate_payment_handler,
                step_payment_handler as step_checkout_payment_handler,
                initiate_order_handler,
                release_handler,
                get_handler as get_checkout_handler,
            };
            use checkout_core::payment::handlers::{
                step_handler,
                callback_handler,
                cancel_handler as cancel_payment_handler,
                get_handler as get_payment_handler,
            };
            use checkout_core::order::handlers::{
                // get_handler as get_order_handler,
                // step_payment_handler as step_order_payment_handler,
                // confirm_handler
                // complete_handler
                // cancel_handler as cancel_order_handler,
                reconcile_handler,
            };

            pub struct Config {
                pub google_application_credentials: String,
                pub payment_callbacks_topic: String,
                pub order_events_topic: String,
                pub bind_address: String,
            }

            pub fn get_config() -> Config {
                Config {
                    google_application_credentials: env::var("GOOGLE_APPLICATION_CREDENTIALS").unwrap(),
                    payment_callbacks_topic: env::var("PAYMENT_CALLBACKS_TOPIC").unwrap(),
                    order_events_topic: env::var("ORDER_EVENTS_TOPIC").unwrap(),
                    bind_address: env::var("BIND_ADDRESS").unwrap(),
                }
            }

            // begin addition

            pub struct MasterContextProvider {
                db_client: DBClient,
                client_context_provider: super::#server_type_name,
            }

            impl MasterContextProvider {
                pub fn new(db_client: DBClient, client_context_provider: super::#server_type_name) -> Self {
                    Self { db_client, client_context_provider }
                }

                pub async fn new_context<#common_lifetime>(&#common_lifetime self) -> Result<MasterContext<#common_lifetime, super::#context_type_name #ty_generics>, Error> {
                    let client_ctx = self.client_context_provider.new_context().await?;
                    Ok(MasterContext::new(client_ctx, &self.db_client))
                }
            }

            // end addition

            pub struct ServerContext {
                pub internal_context: InternalContext,
                pub context_provider: Arc<MasterContextProvider>,
            }

            #[derive(Debug)]
            pub struct IdWrapper(String);

            impl cloud_pubsub::FromPubSubMessage for IdWrapper {
                fn from(message: cloud_pubsub::EncodedMessage) -> Result<Self, cloud_pubsub::error::Error> {
                    match message.decode() {
                        Ok(bytes) => {
                            let val: serde_json::Value = serde_json::from_slice(&bytes).unwrap();
                            Ok(IdWrapper(val.as_str().unwrap().to_string()))
                        }
                        Err(e) => Err(cloud_pubsub::error::Error::from(e)),
                    }
                }
            }

            pub fn process_payment_callbacks(
                internal_ctx: InternalContext,
                context_provider: Arc<MasterContextProvider>,
            ) {
                task::spawn(async move {
                    let subscription = internal_ctx.payment_callbacks_subscription.clone();

                    while subscription.client().is_running() {
                        println!("pulling payment processing messages");

                        let pull = subscription.get_messages::<CallbackArgs<<super::#context_type_name #ty_static_generics as PaymentProcessor>::ProcessArgs>>();

                        match timeout(Duration::from_secs(30), pull).await {
                            Ok(result) => match result {
                                Ok(messages) => {
                                    for (result, ack_id) in messages {
                                        match result {
                                            Ok(message) => {
                                                println!("Received CallbackArgs message");
                                                match context_provider.new_context().await {
                                                    Ok(mut ctx) => {
                                                        if !ctx.start_transaction().await.is_ok() {
                                                            continue;
                                                        }
                                                        let processing_result = callback_handler(&internal_ctx, &mut ctx, &message.correlation_id, message.process_args).await;
                                                        if !processing_result.is_ok() {
                                                            if !ctx.abort_transaction().await.is_ok() {
                                                                // just log the failure to abort tx
                                                                eprintln!("failed to abort tx");
                                                            }

                                                            let processing_err = processing_result.err().unwrap();
                                                            if &processing_err.code != "SERVER_ERROR" {
                                                                subscription.acknowledge_messages(vec![ack_id]).await;
                                                            }

                                                            continue;
                                                        }
                                                        if !ctx.commit_transaction().await.is_ok() {
                                                            continue;
                                                        }
                                                        subscription.acknowledge_messages(vec![ack_id]).await;
                                                    }
                                                    Err(_) => {}
                                                }
                                            }
                                            Err(e) => eprintln!("Failed to parse message: {}", e),
                                        }
                                    }
                                }
                                Err(e) => eprintln!("Failed to pull PubSub messages: {}", e)
                            },
                            Err(_) => eprintln!("no response in 30 seconds")
                        }
                    }
                });
            }

            pub fn process_order_events(
                internal_ctx: InternalContext,
                context_provider: Arc<MasterContextProvider>,
            ) {
                task::spawn(async move {
                    let subscription = internal_ctx.order_events_subscription.clone();

                    while subscription.client().is_running() {
                        println!("pulling order events messages");

                        let pull = subscription.get_messages::<LookupKey>();

                        match timeout(Duration::from_secs(30), pull).await {
                            Ok(result) => match result {
                                Ok(messages) => {
                                    for (result, ack_id) in messages {
                                        match result {
                                            Ok(message) => {
                                                println!("Received: {:?}", message);
                                                match context_provider.new_context().await {
                                                    Ok(mut ctx) => {
                                                        if !ctx.start_transaction().await.is_ok() {
                                                            continue;
                                                        }
                                                        let reconciliation_result = reconcile_handler(&mut ctx, &message).await;
                                                        if !reconciliation_result.is_ok() {
                                                            if !ctx.abort_transaction().await.is_ok() {
                                                                // just log the failure to abort tx
                                                                eprintln!("failed to commit tx");
                                                            }
                                                            continue;
                                                        }
                                                        if !ctx.commit_transaction().await.is_ok() {
                                                            continue;
                                                        }
                                                        subscription.acknowledge_messages(vec![ack_id]).await;
                                                    }
                                                    Err(_) => {}
                                                }
                                            }
                                            Err(e) => eprintln!("Failed to parse message: {}", e),
                                        }
                                    }
                                }
                                Err(e) => eprintln!("Failed to pull PubSub messages: {}", e)
                            },
                            Err(_) => eprintln!("no response in 30 seconds")
                        }
                    }
                });
            }

            #[derive(Serialize, Deserialize, JsonSchema)]
            pub struct IdParam {
                id: String
            }

            #[derive(Serialize, Deserialize, JsonSchema)]
            pub struct CorrelationIdParam {
                correlation_id: String
            }

            #[derive(Serialize, Deserialize, JsonSchema)]
            pub struct CreateCheckoutParams {
                pub currency: Currency,
                pub contact: Option<Contact>,
                pub shipping_address: Option<Address>,
            }

            #[derive(Serialize, Deserialize, JsonSchema)]
            pub struct AddOrRemoveItemParams {
                pub sku: String,
                pub quantity: u64,
            }

            #[endpoint {
                method = POST,
                path = "/checkout",
            }]
            pub async fn create_checkout(
                req_ctx: Arc<RequestContext<ServerContext>>,
                body: TypedBody<CreateCheckoutParams>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let params = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match create_handler(&mut ctx, params.currency.clone(), params.contact.clone(), params.shipping_address.clone()).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = GET,
                path = "/checkout/{id}",
            }]
            pub async fn get_checkout(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match get_checkout_handler(&mut ctx, &path_params.id).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/add_item",
            }]
            pub async fn add_checkout_item(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<AddOrRemoveItemParams>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let body_params = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match add_item_handler(&mut ctx, &path_params.id, body_params.sku.clone(), body_params.quantity).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/remove_item",
            }]
            pub async fn remove_checkout_item(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<AddOrRemoveItemParams>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let body_params = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match remove_item_handler(&mut ctx, &path_params.id, body_params.sku.clone(), body_params.quantity).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/confirm_items",
            }]
            pub async fn confirm_checkout_items(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();

                match confirm_items_handler(&mut ctx, &path_params.id).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/update_contact",
            }]
            pub async fn update_checkout_contact(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<Contact>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let contact = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match update_contact_handler(&mut ctx, &path_params.id, contact).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/update_shipping_address",
            }]
            pub async fn update_checkout_shipping_address(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<Address>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let shipping_address = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match update_shipping_address_handler(&mut ctx, &path_params.id, shipping_address).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/update_fulfillment",
            }]
            pub async fn update_checkout_fulfillment(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<FulfillmentSelection>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let fulfillment_selection = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match update_fulfillment_handler(&mut ctx, &path_params.id, fulfillment_selection).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/initiate_payment",
            }]
            pub async fn initiate_checkout_payment(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<<super::#context_type_name #ty_static_generics as PaymentProcessor>::InitArgs>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let args = body.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match initiate_payment_handler(&mut ctx, &path_params.id, args).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/step_payment",
            }]
            pub async fn step_checkout_payment(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<<super::#context_type_name #ty_static_generics as PaymentProcessor>::ProcessArgs>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let args = body.into_inner();
                let internal_ctx = req_ctx
                    .context()
                    .internal_context
                    .clone();

                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match step_checkout_payment_handler(&internal_ctx, &mut ctx, &path_params.id, args).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/initiate_order",
            }]
            pub async fn initiate_checkout_order(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<<super::#context_type_name #ty_static_generics as PaymentProcessor>::InitArgs>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let args = body.into_inner();
                let internal_ctx = req_ctx
                    .context()
                    .internal_context
                    .clone();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match initiate_order_handler(&internal_ctx, &mut ctx, &path_params.id, args).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/checkout/{id}/release",
            }]
            pub async fn release_checkout(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
            ) -> Result<HttpResponseOk<Checkout<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>>, HttpError> {
                let path_params = path.into_inner();
                let internal_ctx = req_ctx
                    .context()
                    .internal_context
                    .clone();

                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match release_handler(&internal_ctx, &mut ctx, &path_params.id).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            // payment related endpoints
            #[endpoint {
                method = GET,
                path = "/payment/{id}",
            }]
            pub async fn get_payment(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
            ) -> Result<HttpResponseOk<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>, HttpError> {
                let path_params = path.into_inner();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match get_payment_handler(&mut ctx, &path_params.id).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/payment/{id}/step",
            }]
            pub async fn step_payment(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
                body: TypedBody<<super::#context_type_name #ty_static_generics as PaymentProcessor>::ProcessArgs>,
            ) -> Result<HttpResponseOk<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>, HttpError> {
                let path_params = path.into_inner();
                let args = body.into_inner();
                let internal_ctx = req_ctx
                    .context()
                    .internal_context
                    .clone();

                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match step_handler(&internal_ctx, &mut ctx, &path_params.id, args).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #[endpoint {
                method = PATCH,
                path = "/payment/{id}/cancel",
            }]
            pub async fn cancel_payment(
                req_ctx: Arc<RequestContext<ServerContext>>,
                path: Path<IdParam>,
            ) -> Result<HttpResponseOk<Payment<
                <super::#context_type_name #ty_static_generics as PaymentProcessor>::Data
            >>, HttpError> {
                let path_params = path.into_inner();
                let internal_ctx = req_ctx
                    .context()
                    .internal_context
                    .clone();
                let mut ctx = req_ctx
                    .context()
                    .context_provider
                    .new_context()
                    .await
                    .unwrap();

                ctx.start_transaction().await.unwrap();
                match cancel_payment_handler(&internal_ctx, &mut ctx, &path_params.id).await {
                    Ok(co) => {
                        ctx.commit_transaction().await.unwrap();
                        Ok(HttpResponseOk(co))
                    }
                    Err(err) => {
                        ctx.abort_transaction().await.unwrap();
                        Err(err.into())
                    }
                }
            }

            #payment_webhook_endpoint_def
        }

        #[async_trait]
        impl #impl_generics checkout_server::Server<#common_lifetime, #context_type_name #ty_generics> for #server_type_name {
            type Context = checkout_server::ServerContext;

            fn api_description() -> checkout_server::ApiDescription<Self::Context> {
                let mut api = checkout_server::ApiDescription::new();
                api.register(checkout_server::create_checkout).unwrap();
                api.register(checkout_server::get_checkout).unwrap();
                api.register(checkout_server::add_checkout_item).unwrap();
                api.register(checkout_server::remove_checkout_item).unwrap();
                api.register(checkout_server::confirm_checkout_items).unwrap();
                api.register(checkout_server::update_checkout_contact).unwrap();
                api.register(checkout_server::update_checkout_shipping_address).unwrap();
                api.register(checkout_server::update_checkout_fulfillment).unwrap();
                api.register(checkout_server::initiate_checkout_payment).unwrap();
                api.register(checkout_server::step_checkout_payment).unwrap();
                api.register(checkout_server::initiate_checkout_order).unwrap();
                api.register(checkout_server::release_checkout).unwrap();
                api.register(checkout_server::get_payment).unwrap();
                api.register(checkout_server::step_payment).unwrap();
                api.register(checkout_server::cancel_payment).unwrap();

                #payment_webhook_endpoint_reg

                api
            }

            async fn start(
                self
            ) -> Result<(), checkout_server::Error> {
                let log = checkout_server::ConfigLogging::StderrTerminal {
                    level: checkout_server::ConfigLoggingLevel::Info,
                }
                .to_logger("minimal-example")
                .map_err(|_| checkout_server::new_server_error("failed to setup logger"))?;

                let api = Self::api_description();
                let config = checkout_server::get_config();
                let internal_context = checkout_server::InternalContext::new(config.google_application_credentials.clone(), config.payment_callbacks_topic.clone(), config.order_events_topic.clone()).await?;

                let db_client = checkout_server::DBClient::new().await.map_err(|_| checkout_server::new_server_error("failed to create database client"))?;
                let context_provider = checkout_server::Arc::new(checkout_server::MasterContextProvider::new(
                    db_client,
                    self,
                ));

                let pubsub_client = internal_context.pubsub_client.clone();
                checkout_server::spawn_token_renew(&pubsub_client, checkout_server::Duration::from_secs(15 * 60));

                checkout_server::process_payment_callbacks(internal_context.clone(), context_provider.clone());
                checkout_server::process_order_events(internal_context.clone(), context_provider.clone());

                checkout_server::HttpServerStarter::new(
                    &checkout_server::ConfigDropshot {
                        bind_address: config.bind_address.parse().unwrap(),
                        request_body_max_bytes: 2048 * 1024,
                    },
                    api,
                    checkout_server::ServerContext {
                        internal_context: internal_context.clone(),
                        context_provider: context_provider.clone(),
                    },
                    &log,
                )
                .map_err(|_| checkout_server::new_server_error("failed to set up api server"))?
                .start()
                .await
                .map_err(|_| checkout_server::new_server_error("failed to start server"))
            }
        }
    };
    gen.into()
}
