use cloud_pubsub;
use serde_json;
use tokio::task;

use crate::context::{Context, ContextProvider};
use crate::internal_context::InternalContext;
use crate::payment::handlers::process_handler;

#[derive(Debug)]
struct IdWrapper(String);
impl cloud_pubsub::FromPubSubMessage for IdWrapper {
    fn from(message: cloud_pubsub::EncodedMessage) -> Result<Self, cloud_pubsub::error::Error> {
        match message.decode() {
            Ok(bytes) => {
                let val: serde_json::Value = serde_json::from_slice(&bytes).unwrap();
                Ok(IdWrapper(val.as_str().unwrap().to_string()))
            }
            Err(e) => Err(cloud_pubsub::error::Error::from(e)),
        }
    }
}

pub fn start_payment_processing<
    C: Context + Send + Sync,
    CP: ContextProvider<'static, C> + Send + Sync,
>(
    internal_ctx: InternalContext,
    context_provider: &'static CP,
) {
    task::spawn(async move {
        let subscription = internal_ctx.payment_processing_subscription.clone();

        while subscription.client().is_running() {
            println!("pulling payment processing messages");

            match subscription.get_messages::<IdWrapper>().await {
                Ok(messages) => {
                    for (result, ack_id) in messages {
                        match result {
                            Ok(message) => {
                                println!("Received: {:?}", message);
                                match context_provider.new_context().await {
                                    Ok(mut ctx) => {
                                        if !ctx.start_transaction().await.is_ok() {
                                            continue;
                                        }
                                        let processing_result =
                                            process_handler(&internal_ctx, &mut ctx, &message.0)
                                                .await;
                                        if !processing_result.is_ok() {
                                            if !ctx.abort_transaction().await.is_ok() {
                                                // just log the failure to abort tx
                                                eprintln!("failed to commit tx");
                                            }
                                            continue;
                                        }
                                        if !ctx.commit_transaction().await.is_ok() {
                                            continue;
                                        }
                                        subscription.acknowledge_messages(vec![ack_id]).await;
                                    }
                                    Err(_) => {}
                                }
                            }
                            Err(e) => eprintln!("Failed to parse message: {}", e),
                        }
                    }
                }
                Err(e) => eprintln!("Failed to pull PubSub messages: {}", e),
            }
        }
    });
}
