use async_trait::async_trait;
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use uuid::Uuid;

use crate::checkout::Checkout;
use crate::context::Context;
use crate::error::{new_application_error, Error};
use crate::internal_context::InternalContext;
use crate::money::Money;
use crate::order::Order;

pub mod handlers;

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct Payment<PrivateData, PublicData, ProcessArgs> {
    pub id: String,
    pub state: PaymentState<ProcessArgs>,
    pub initiator_id: InitiatorId,
    pub charge_amount: Money,
    pub data: PaymentData<PrivateData, PublicData>,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum PaymentState<ProcessArgs> {
    Idle,
    Processing(ProcessArgs),
    Succeeded,
    Cancelled,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum InitiatorId {
    Checkout(String),
    Order(String),
}

pub enum Initiator<'a> {
    Checkout(&'a mut Checkout),
    Order(&'a mut Order),
}

#[async_trait]
pub trait PaymentProcessor {
    type ProcessArgs: Sync;
    type PrivateData: Sync;
    type PublicData: Sync;

    async fn initiate_payment(
        &mut self,
        _payment: &Payment<Self::PrivateData, Self::PublicData, Self::ProcessArgs>,
        _args: &Self::ProcessArgs,
    ) -> Result<PaymentData<Self::PrivateData, Self::PublicData>, Error> {
        Ok(PaymentData {
            correlation_id: None,
            private_data: None,
            public_data: None,
        })
    }

    async fn process_payment(
        &mut self,
        payment: &Payment<Self::PrivateData, Self::PublicData, Self::ProcessArgs>,
        args: &Self::ProcessArgs,
    ) -> Result<ProcessingResult<Self::PrivateData, Self::PublicData>, Error>;

    // this is called synchronously and only once per revert request
    async fn cancel_payment(
        &mut self,
        payment: &Payment<Self::PrivateData, Self::PublicData, Self::ProcessArgs>,
    ) -> Result<CancellationResult, Error>;
}

#[async_trait]
pub trait PaymentStore: PaymentProcessor {
    async fn create_payment(
        &mut self,
        payment: &Payment<
            <Self as PaymentProcessor>::PrivateData,
            <Self as PaymentProcessor>::PublicData,
            <Self as PaymentProcessor>::ProcessArgs,
        >,
    ) -> Result<(), Error>;

    async fn update_payment(
        &mut self,
        payment: &Payment<
            <Self as PaymentProcessor>::PrivateData,
            <Self as PaymentProcessor>::PublicData,
            <Self as PaymentProcessor>::ProcessArgs,
        >,
    ) -> Result<(), Error>;

    async fn get_payment(
        &mut self,
        id: &str,
    ) -> Result<
        Option<
            Payment<
                <Self as PaymentProcessor>::PrivateData,
                <Self as PaymentProcessor>::PublicData,
                <Self as PaymentProcessor>::ProcessArgs,
            >,
        >,
        Error,
    >;

    async fn get_payment_for_update(
        &mut self,
        id: &str,
    ) -> Result<
        Option<
            Payment<
                <Self as PaymentProcessor>::PrivateData,
                <Self as PaymentProcessor>::PublicData,
                <Self as PaymentProcessor>::ProcessArgs,
            >,
        >,
        Error,
    >;

    async fn get_payment_for_update_by_correlation_id(
        &mut self,
        _id: &str,
    ) -> Result<
        Option<
            Payment<
                <Self as PaymentProcessor>::PrivateData,
                <Self as PaymentProcessor>::PublicData,
                <Self as PaymentProcessor>::ProcessArgs,
            >,
        >,
        Error,
    > {
        Ok(None)
    }
}

pub struct ProcessingResult<PrivateData, PublicData> {
    pub state: InternalPaymentState,
    pub data: PaymentData<PrivateData, PublicData>,
}

pub struct CancellationResult {
    pub success: bool,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct PaymentData<PrivateData, PublicData> {
    pub correlation_id: Option<String>,
    pub private_data: Option<PrivateData>,
    pub public_data: Option<PublicData>,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum InternalPaymentState {
    Incomplete,
    Succeeded,
    Failed,
    Cancelled,
}

// constructor
async fn new<'a, C: Context + Send>(
    ctx: &mut C,
    initiator: Initiator<'a>,
    charge_amount: Money,
    args: <C as PaymentProcessor>::ProcessArgs,
) -> Result<
    Payment<
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    Error,
> {
    let mut payment = Payment {
        id: Uuid::new_v4().to_string(),
        state: PaymentState::Idle,
        initiator_id: match &initiator {
            Initiator::Checkout(co) => InitiatorId::Checkout(co.id.clone()),
            Initiator::Order(od) => InitiatorId::Order(od.id.clone()),
        },
        charge_amount,
        data: PaymentData {
            correlation_id: None,
            private_data: None,
            public_data: None,
        },
    };

    payment.data = ctx.initiate_payment(&payment, &args).await?;
    ctx.create_payment(&payment).await?;

    Ok(payment)
}

// public methods
pub async fn queue_for_processing<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    payment: &mut Payment<
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    args: <C as PaymentProcessor>::ProcessArgs,
) -> Result<(), Error> {
    match &payment.state {
        PaymentState::Idle => {
            payment.state = PaymentState::Processing(args);
            ctx.update_payment(payment).await?;

            internal_ctx
                .payment_processing_topic
                .clone()
                .publish(&payment.id)
                .await
                .map_err(|_| {
                    new_application_error(
                        "UPSTREAM_ERROR",
                        "couldn't send message to payment processing topic",
                    )
                })?;

            Ok(())
        }
        PaymentState::Processing(_) => Err(new_application_error(
            "PAYMENT_PROCESSING",
            "processing is in progress",
        )),
        PaymentState::Succeeded | PaymentState::Cancelled => Err(new_application_error(
            "PAYMENT_COMPLETED",
            "the payment has already completed",
        )),
    }
}

pub async fn process<'a, C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    payment: &mut Payment<
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    initiator: Initiator<'a>,
) -> Result<(), Error> {
    match &payment.state {
        PaymentState::Processing(args) => {
            let result = ctx.process_payment(payment, args).await?;
            payment.data = result.data;
            match result.state {
                InternalPaymentState::Incomplete | InternalPaymentState::Failed => {
                    payment.state = PaymentState::Idle
                }
                InternalPaymentState::Succeeded => payment.state = PaymentState::Succeeded,
                InternalPaymentState::Cancelled => payment.state = PaymentState::Cancelled,
            }

            ctx.update_payment(payment).await?;

            match payment.state {
                PaymentState::Succeeded => match initiator {
                    Initiator::Checkout(co) => co.complete(internal_ctx, ctx).await,
                    Initiator::Order(od) => od.prompt_confirmation(internal_ctx, ctx).await,
                },
                PaymentState::Cancelled => match initiator {
                    Initiator::Checkout(co) => co.revert(ctx).await,
                    Initiator::Order(od) => od.cancel(internal_ctx, ctx).await,
                },
                _ => Ok(()),
            }
        }
        _ => Ok(()),
    }
}

pub async fn cancel<'a, C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    payment: &mut Payment<
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    initiator: Initiator<'a>,
) -> Result<(), Error> {
    match &payment.state {
        PaymentState::Cancelled => Ok(()),
        PaymentState::Idle => {
            let cancellation_result = ctx.cancel_payment(payment).await?;
            if !cancellation_result.success {
                return Err(new_application_error(
                    "CANCELLATION_UNAVAILABLE",
                    "the payment can no longer be cancelled",
                ));
            }

            payment.state = PaymentState::Cancelled;
            ctx.update_payment(payment).await?;

            match initiator {
                Initiator::Checkout(co) => co.revert(ctx).await,
                Initiator::Order(od) => od.cancel(internal_ctx, ctx).await,
            }
        }
        PaymentState::Processing(_) => Err(new_application_error(
            "PAYMENT_PROCESSING",
            "the payment is processing",
        )),
        PaymentState::Succeeded => Err(new_application_error(
            "PAYMENT_COMPLETED",
            "the payment has already completed",
        )),
    }
}
