use crate::context::Context;
use crate::error::new_not_found_error;
use crate::error::Error;
use crate::internal_context::InternalContext;
use crate::payment::cancel as cancel_payment;
use crate::payment::Initiator;
use crate::payment::Payment;
use crate::payment::PaymentProcessor;

use super::Order;
use super::OrderState;

pub async fn confirm_handler<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    order_id: &str,
) -> Result<Order, Error> {
    let maybe_od = ctx.get_order_for_update(&order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let mut od = maybe_od.unwrap();
    od.confirm(internal_ctx, ctx).await?;

    Ok(od)
}

pub async fn cancel_handler<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    order_id: &str,
) -> Result<Order, Error> {
    // first retrieve the associated payment for update to avoid deadlock
    let maybe_payment = get_order_payment_for_update(ctx, order_id).await?;

    let maybe_od = ctx.get_order_for_update(&order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let mut od = maybe_od.unwrap();

    if let OrderState::PaymentInProgress = od.state {
        if maybe_payment.is_none() {
            return Err(new_not_found_error(
                "the requested payment could not be found",
            ));
        }

        let mut payment = maybe_payment.unwrap();
        cancel_payment(internal_ctx, ctx, &mut payment, Initiator::Order(&mut od)).await?;
    } else {
        od.cancel(internal_ctx, ctx).await?;
    }

    Ok(od)
}

async fn get_order_payment_for_update<C: Context + Send>(
    ctx: &mut C,
    order_id: &str,
) -> Result<
    Option<
        Payment<
            <C as PaymentProcessor>::PrivateData,
            <C as PaymentProcessor>::PublicData,
            <C as PaymentProcessor>::ProcessArgs,
        >,
    >,
    Error,
> {
    let maybe_od = ctx.get_order(order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let od = maybe_od.unwrap();
    ctx.get_payment_for_update(od.payment_id.as_ref().map_or("", |payment_id| payment_id))
        .await
}
