use async_trait::async_trait;
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use uuid::Uuid;

use crate::{
    checkout::Checkout,
    context::Context,
    customer::{Address, Contact},
    error::{new_invalid_state_error, Error},
    invoice::Invoice,
    item::Item,
    payment::PaymentProcessor,
    payment::{handlers as payment_handlers, Initiator},
    shipping::FulfillmentType,
};

pub mod handlers;

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct Order {
    pub id: String,
    pub state: OrderState,
    pub reconciling_state: bool,
    pub contact: Contact,
    pub shipping_address: Address,
    pub items: Vec<Item>,
    pub fulfillment_type: FulfillmentType,
    pub invoice: Invoice,
    pub payment_id: Option<String>,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum OrderState {
    PaymentInProgress, // initial state if the order is created via a checkout without payment; transitions to PendingConfirmation or Cancelled
    PendingConfirmation, // initial state if the order is created via a checkout with payment; transitions to Confirmed or Cancelled
    Confirmed, // can only enter this state from PendingConfirmation; transitions to Fulfilled or Cancelled
    Fulfilled, // a terminal state
    Cancelled, // a terminal state
}

#[async_trait]
pub trait OrderStore {
    async fn create_order(&mut self, order: &Order) -> Result<(), Error>;
    async fn update_order(&mut self, order: &Order) -> Result<(), Error>;
    async fn get_order_for_update(&mut self, id: &str) -> Result<Option<Order>, Error>;
    async fn get_order(&mut self, id: &str) -> Result<Option<Order>, Error>;
}

#[async_trait]
pub trait OrderEvents {
    async fn on_pending_payment(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_pending_confirmation(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_confirmed(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_fulfilled(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_cancelled(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }
}

impl Order {
    pub async fn new_from_checkout<C: Context + Send>(
        ctx: &mut C,
        checkout: &Checkout,
    ) -> Result<Self, Error> {
        let order = Order {
            id: Uuid::new_v4().to_string(),
            reconciling_state: false,
            state: OrderState::PaymentInProgress,
            contact: checkout.contact.as_ref().unwrap().clone(),
            shipping_address: checkout.shipping_address.as_ref().unwrap().clone(),
            items: checkout.items.clone(),
            fulfillment_type: checkout.fulfillment_type.as_ref().unwrap().clone(),
            invoice: checkout.invoice.as_ref().unwrap().clone(),
            payment_id: checkout.payment_id.clone(),
        };

        ctx.create_order(&order).await?;
        Ok(order)
    }

    pub async fn new_from_checkout_missing_payment<C: Context + Send>(
        ctx: &mut C,
        checkout: &Checkout,
        payment_args: Option<<C as PaymentProcessor>::ProcessArgs>,
    ) -> Result<Self, Error> {
        let mut order = Order {
            id: Uuid::new_v4().to_string(),
            reconciling_state: false,
            state: OrderState::PaymentInProgress,
            contact: checkout.contact.as_ref().unwrap().clone(),
            shipping_address: checkout.shipping_address.as_ref().unwrap().clone(),
            items: checkout.items.clone(),
            fulfillment_type: checkout.fulfillment_type.as_ref().unwrap().clone(),
            invoice: checkout.invoice.as_ref().unwrap().clone(),
            payment_id: None,
        };

        let charge_amount = order.invoice.initial_charge_amount.clone();
        let payment = payment_handlers::create_handler(
            ctx,
            Initiator::Order(&mut order),
            charge_amount,
            payment_args,
        )
        .await?;

        order.payment_id = Some(payment.id.clone());

        ctx.create_order(&order).await?;
        Ok(order)
    }
}

impl Order {
    fn enter_pending_confirmation_state(&mut self) {
        self.state = OrderState::PendingConfirmation;
    }

    fn enter_confirmed_state(&mut self) {
        self.state = OrderState::Confirmed;
    }

    fn enter_cancelled_state(&mut self) {
        self.state = OrderState::Cancelled;
    }
}

impl Order {
    fn ensure_pending_confirmation_state(&mut self) -> Result<(), Error> {
        match self.state {
            OrderState::PendingConfirmation => Ok(()),
            OrderState::PaymentInProgress => Ok(self.enter_pending_confirmation_state()),
            _ => Err(new_invalid_state_error(
                "should be in the PaymentInProgress state",
            )),
        }
    }

    fn ensure_confirmed_state(&mut self) -> Result<(), Error> {
        match self.state {
            OrderState::Confirmed => Ok(()),
            OrderState::PendingConfirmation => Ok(self.enter_confirmed_state()),
            _ => Err(new_invalid_state_error(
                "should be in the PendingConfirmation state",
            )),
        }
    }

    fn ensure_cancelled_state(&mut self) -> Result<(), Error> {
        match self.state {
            OrderState::Cancelled => Ok(()),
            OrderState::PaymentInProgress
            | OrderState::PendingConfirmation
            | OrderState::Confirmed => Ok(self.enter_cancelled_state()),
            _ => Err(new_invalid_state_error(
                "should be in the PaymentInProgress, PendingConfirmation or Confirmed states",
            )),
        }
    }
}

impl Order {
    pub async fn prompt_confirmation<C: Context + Send>(
        &mut self,
        ctx: &mut C,
    ) -> Result<(), Error> {
        self.ensure_pending_confirmation_state()?;
        ctx.update_order(self).await
    }

    pub async fn confirm<C: Context + Send>(&mut self, ctx: &mut C) -> Result<(), Error> {
        self.ensure_confirmed_state()?;
        ctx.update_order(self).await
    }

    pub async fn cancel<C: Context + Send>(&mut self, ctx: &mut C) -> Result<(), Error> {
        self.ensure_cancelled_state()?;
        ctx.update_order(self).await
    }
}
