use crate::context::Context;
use crate::error::new_not_found_error;
use crate::error::Error;
use crate::internal_context::InternalContext;
use crate::payment::cancel as cancel_payment;
use crate::payment::Initiator;

use super::Order;
use super::OrderState;

pub async fn confirm_handler<C: Context + Send>(
    internal_context: &InternalContext,
    ctx: &mut C,
    order_id: &str,
) -> Result<Order, Error> {
    let maybe_od = ctx.get_order_for_update(&order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let mut od = maybe_od.unwrap();
    od.confirm(ctx).await?;

    Ok(od)
}

pub async fn cancel_handler<C: Context + Send>(
    internal_context: &InternalContext,
    ctx: &mut C,
    order_id: &str,
) -> Result<Order, Error> {
    let maybe_od = ctx.get_order_for_update(&order_id).await?;
    if maybe_od.is_none() {
        return Err(new_not_found_error(
            "the requested order could not be found",
        ));
    }

    let mut od = maybe_od.unwrap();

    if let OrderState::PaymentInProgress = od.state {
        let maybe_payment = ctx
            .get_payment_for_update(od.payment_id.as_ref().map_or("", |payment_id| payment_id))
            .await?;
        if maybe_payment.is_none() {
            return Err(new_not_found_error(
                "the requested payment could not be found",
            ));
        }

        let mut payment = maybe_payment.unwrap();
        cancel_payment(ctx, &mut payment, Initiator::Order(&mut od)).await?;
    } else {
        od.cancel(ctx).await?;
    }

    Ok(od)
}
