use async_trait::async_trait;
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

use crate::{
    customer::{Address, Contact},
    error::Error,
    invoice::Invoice,
    item::Item,
    shipping::FulfillmentType,
};

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum OrderState {
    PendingPayment, // initial state if the order is created manually; transitions to PendingConfirmation, Confirmed or Cancelled
    PendingConfirmation, // a possible initial state if the order is created via checkout completion; transitions to Confirmed or Cancelled
    Confirmed, // a possible initial state if the order is created via checkout completion; transitions to Shipped or Cancelled
    Shipped,   // transitions to Fulfilled or Returned
    Fulfilled, // transitions to Returned
    Returned,  // a terminal state
    Cancelled, // a terminal state
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct Order {
    pub id: String,
    pub state: OrderState,
    pub contact: Contact,
    pub shipping_address: Address,
    pub items: Vec<Item>,
    pub fulfillment_type: Option<FulfillmentType>,
    pub invoice: Invoice,
    pub payment_id: String,
}

#[async_trait]
pub trait OrderStore {
    async fn create_order(&mut self, order: &Order) -> Result<(), Error>;
    async fn update_order(&mut self, order: &Order) -> Result<(), Error>;
    async fn get_order_for_update(&mut self, id: &str) -> Result<Option<Order>, Error>;
    async fn get_order(&mut self, id: &str) -> Result<Option<Order>, Error>;
}

#[async_trait]
pub trait OrderEvents {
    async fn on_pending_payment(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_pending_confirmation(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_confirmed(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_shipped(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_fulfilled(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_returned(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }

    async fn on_cancelled(&mut self, _order: &Order) -> Result<(), Error> {
        Ok(())
    }
}
