use crate::checkout::Checkout;
use crate::context::Context;
use crate::customer::{Address, Contact};
use crate::error::{new_not_found_error, Error};
use crate::money::Currency;
use crate::payment::PaymentProcessor;
use crate::shipping::FulfillmentTypeSelection;

pub async fn create_handler<T: Context + Send>(
    ctx: &mut T,
    currency: Currency,
    contact: Option<Contact>,
    shipping_address: Option<Address>,
) -> Result<Checkout, Error> {
    Checkout::new(ctx, currency, contact, shipping_address).await
}

pub async fn add_item_handler<T: Context + Send>(
    ctx: &mut T,
    checkout_id: &str,
    sku: String,
    quantity: u64,
) -> Result<Checkout, Error> {
    let maybe_co = ctx.get_checkout_for_update(&checkout_id).await?;
    if maybe_co.is_none() {
        return Err(new_not_found_error(
            "the requested checkout could not be found",
        ));
    }

    let mut co = maybe_co.unwrap();

    co.add_item(ctx, sku, quantity).await?;

    Ok(co)
}

pub async fn remove_item_handler<T: Context + Send>(
    ctx: &mut T,
    checkout_id: &str,
    sku: String,
    quantity: u64,
) -> Result<Checkout, Error> {
    let maybe_co = ctx.get_checkout_for_update(&checkout_id).await?;
    if maybe_co.is_none() {
        return Err(new_not_found_error(
            "the requested checkout could not be found",
        ));
    }

    let mut co = maybe_co.unwrap();

    co.remove_item(ctx, sku, quantity).await?;

    Ok(co)
}

pub async fn confirm_items_handler<T: Context + Send>(
    ctx: &mut T,
    checkout_id: &str,
) -> Result<Checkout, Error> {
    let maybe_co = ctx.get_checkout_for_update(&checkout_id).await?;
    if maybe_co.is_none() {
        return Err(new_not_found_error(
            "the requested checkout could not be found",
        ));
    }

    let mut co = maybe_co.unwrap();

    co.confirm_items(ctx).await?;

    Ok(co)
}

pub async fn update_contact_handler<T: Context + Send>(
    ctx: &mut T,
    checkout_id: &str,
    contact: Contact,
) -> Result<Checkout, Error> {
    let maybe_co = ctx.get_checkout_for_update(&checkout_id).await?;
    if maybe_co.is_none() {
        return Err(new_not_found_error(
            "the requested checkout could not be found",
        ));
    }

    let mut co = maybe_co.unwrap();

    co.update_contact(ctx, contact).await?;

    Ok(co)
}

pub async fn update_shipping_address_handler<T: Context + Send>(
    ctx: &mut T,
    checkout_id: &str,
    shipping_address: Address,
) -> Result<Checkout, Error> {
    let maybe_co = ctx.get_checkout_for_update(&checkout_id).await?;
    if maybe_co.is_none() {
        return Err(new_not_found_error(
            "the requested checkout could not be found",
        ));
    }

    let mut co = maybe_co.unwrap();

    co.update_shipping_address(ctx, shipping_address).await?;

    Ok(co)
}

pub async fn update_fulfillment_type_handler<T: Context + Send>(
    ctx: &mut T,
    checkout_id: &str,
    fulfillment_type: FulfillmentTypeSelection,
) -> Result<Checkout, Error> {
    let maybe_co = ctx.get_checkout_for_update(&checkout_id).await?;
    if maybe_co.is_none() {
        return Err(new_not_found_error(
            "the requested checkout could not be found",
        ));
    }

    let mut co = maybe_co.unwrap();

    co.update_fulfillment_type(ctx, fulfillment_type).await?;

    Ok(co)
}

pub async fn initiate_payment_handler<T: Context + Send>(
    ctx: &mut T,
    checkout_id: &str,
    args: <T as PaymentProcessor>::InitiateArgs,
) -> Result<Checkout, Error> {
    let maybe_co = ctx.get_checkout_for_update(&checkout_id).await?;
    if maybe_co.is_none() {
        return Err(new_not_found_error(
            "the requested checkout could not be found",
        ));
    }

    let mut co = maybe_co.unwrap();

    co.initiate_payment(ctx, args).await?;

    Ok(co)
}
