use crate::context::Context;
use crate::error::{new_not_found_error, Error};
use crate::internal_context::InternalContext;
use crate::money::Money;

use super::{cancel, new, process, queue_for_processing, Payment, PaymentProcessor};

pub async fn create_handler<C: Context + Send>(
    ctx: &mut C,
    checkout_id: String,
    charge_amount: Money,
    args: <C as PaymentProcessor>::InitiateArgs,
) -> Result<
    Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    Error,
> {
    new(ctx, checkout_id, charge_amount, args).await
}

pub async fn step_handler<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    id: &str,
    args: <C as PaymentProcessor>::ProcessArgs,
) -> Result<
    Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    Error,
> {
    let maybe_payment = ctx.get_payment_for_update(id).await?;
    if maybe_payment.is_none() {
        return Err(new_not_found_error("the payment could not be found"));
    }
    let mut payment = maybe_payment.unwrap();

    queue_for_processing(internal_ctx, ctx, &mut payment, args).await?;

    Ok(payment)
}

pub async fn callback_handler<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    correlation_id: &<C as PaymentProcessor>::CorrelationID,
    args: <C as PaymentProcessor>::ProcessArgs,
) -> Result<
    Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    Error,
> {
    let maybe_payment = ctx
        .get_payment_for_update_by_correlation_id(correlation_id)
        .await?;
    if maybe_payment.is_none() {
        return Err(new_not_found_error("the payment could not be found"));
    }
    let mut payment = maybe_payment.unwrap();

    queue_for_processing(internal_ctx, ctx, &mut payment, args).await?;

    Ok(payment)
}

pub async fn process_handler<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    id: &str,
) -> Result<(), Error> {
    let maybe_payment = ctx.get_payment_for_update(id).await?;
    if maybe_payment.is_none() {
        // this isn't supposed to happen. maybe log the error and move on
        return Ok(());
    }

    let mut payment = maybe_payment.unwrap();

    let maybe_checkout = ctx.get_checkout_for_update(&payment.checkout_id).await?;
    if maybe_checkout.is_none() {
        // this isn't supposed to happen. maybe log the error and move on
        return Ok(());
    }

    let mut checkout = maybe_checkout.unwrap();

    process(internal_ctx, ctx, &mut payment, &mut checkout).await?;

    Ok(())
}

pub async fn cancel_handler<C: Context + Send>(
    ctx: &mut C,
    id: &str,
) -> Result<
    Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    Error,
> {
    let maybe_payment = ctx.get_payment_for_update(id).await?;
    if maybe_payment.is_none() {
        return Err(new_not_found_error("the payment could not be found"));
    }
    let mut payment = maybe_payment.unwrap();

    let maybe_checkout = ctx.get_checkout_for_update(&payment.checkout_id).await?;
    if maybe_checkout.is_none() {
        return Err(new_not_found_error(
            "the associated checkout could not be found",
        ));
    }

    let mut checkout = maybe_checkout.unwrap();

    cancel(ctx, &mut payment, &mut checkout).await?;

    Ok(payment)
}
