use cloud_pubsub;
use std::sync::Arc;

use crate::context::{Context, ContextProvider};

#[derive(Debug)]
struct Message(String);

impl cloud_pubsub::FromPubSubMessage for Message {
    fn from(message: cloud_pubsub::EncodedMessage) -> Result<Self, cloud_pubsub::error::Error> {
        match message.decode() {
            Ok(bytes) => Ok(Message(String::from_utf8_lossy(&bytes).into_owned())),
            Err(e) => Err(cloud_pubsub::error::Error::from(e)),
        }
    }
}

pub async fn start_payment_processing<'a, C: Context, CP: ContextProvider<'a, C>>(
    client: Arc<cloud_pubsub::Client>,
    subscription_name: String,
    context_provider: Arc<CP>,
) {
    let subscription = client.subscribe(subscription_name);

    while subscription.client().is_running() {
        match subscription.get_messages::<Message>().await {
            Ok(messages) => {
                for (result, ack_id) in messages {
                    match result {
                        Ok(message) => {
                            println!("Received: {:?}", message);
                            subscription.acknowledge_messages(vec![ack_id]).await;
                        }
                        Err(e) => eprintln!("Failed to parse message: {}", e),
                    }
                }
            }
            Err(e) => eprintln!("Failed to pull PubSub messages: {}", e),
        }
    }
}

pub async fn start_checkout_completion(client: Arc<cloud_pubsub::Client>) {}
