use async_trait::async_trait;
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use uuid::Uuid;

use crate::checkout::Checkout;
use crate::context::Context;
use crate::error::{new_application_error, Error};
use crate::internal_context::InternalContext;
use crate::money::Money;

pub mod handlers;

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct Payment<CorrelationID, PrivateData, PublicData, ProcessArgs> {
    pub id: String,
    pub state: PaymentState<ProcessArgs>,
    pub checkout_id: String,
    pub charge_amount: Money,
    pub data: PaymentData<CorrelationID, PrivateData, PublicData>,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum PaymentState<ProcessArgs> {
    Idle,
    Processing(ProcessArgs),
    Succeeded,
    Cancelled,
}

#[async_trait]
pub trait PaymentProcessor {
    type CorrelationID: Sync + PartialEq;
    type InitiateArgs: Sync;
    type ProcessArgs: Sync;
    type PrivateData: Sync;
    type PublicData: Sync;

    // this is called synchronously and only once per initiate_payment request
    async fn initiate_payment(
        &mut self,
        _args: &Self::InitiateArgs,
    ) -> Result<PaymentData<Self::CorrelationID, Self::PrivateData, Self::PublicData>, Error> {
        Ok(PaymentData {
            correlation_id: None,
            private_data: None,
            public_data: None,
        })
    }

    async fn process_payment(
        &mut self,
        payment: &Payment<
            Self::CorrelationID,
            Self::PrivateData,
            Self::PublicData,
            Self::ProcessArgs,
        >,
        args: &Self::ProcessArgs,
    ) -> Result<ProcessingResult<Self::CorrelationID, Self::PrivateData, Self::PublicData>, Error>;

    // this is called synchronously and only once per revert request
    async fn revert(
        &mut self,
        payment: &Payment<
            Self::CorrelationID,
            Self::PrivateData,
            Self::PublicData,
            Self::ProcessArgs,
        >,
    ) -> Result<CancellationResult, Error>;
}

#[async_trait]
pub trait PaymentStore: PaymentProcessor {
    async fn create_payment(
        &mut self,
        payment: &Payment<
            <Self as PaymentProcessor>::CorrelationID,
            <Self as PaymentProcessor>::PrivateData,
            <Self as PaymentProcessor>::PublicData,
            <Self as PaymentProcessor>::ProcessArgs,
        >,
    ) -> Result<(), Error>;

    async fn update_payment(
        &mut self,
        payment: &Payment<
            <Self as PaymentProcessor>::CorrelationID,
            <Self as PaymentProcessor>::PrivateData,
            <Self as PaymentProcessor>::PublicData,
            <Self as PaymentProcessor>::ProcessArgs,
        >,
    ) -> Result<(), Error>;

    async fn get_payment(
        &mut self,
        id: &str,
    ) -> Result<
        Option<
            Payment<
                <Self as PaymentProcessor>::CorrelationID,
                <Self as PaymentProcessor>::PrivateData,
                <Self as PaymentProcessor>::PublicData,
                <Self as PaymentProcessor>::ProcessArgs,
            >,
        >,
        Error,
    >;

    async fn get_payment_for_update(
        &mut self,
        id: &str,
    ) -> Result<
        Option<
            Payment<
                <Self as PaymentProcessor>::CorrelationID,
                <Self as PaymentProcessor>::PrivateData,
                <Self as PaymentProcessor>::PublicData,
                <Self as PaymentProcessor>::ProcessArgs,
            >,
        >,
        Error,
    >;

    async fn get_payment_for_update_by_correlation_id(
        &mut self,
        _id: &<Self as PaymentProcessor>::CorrelationID,
    ) -> Result<
        Option<
            Payment<
                <Self as PaymentProcessor>::CorrelationID,
                <Self as PaymentProcessor>::PrivateData,
                <Self as PaymentProcessor>::PublicData,
                <Self as PaymentProcessor>::ProcessArgs,
            >,
        >,
        Error,
    > {
        Ok(None)
    }
}

pub struct ProcessingResult<CorrelationID, PrivateData, PublicData> {
    pub state: InternalPaymentState,
    pub data: PaymentData<CorrelationID, PrivateData, PublicData>,
}

pub struct CancellationResult {
    pub success: bool,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub struct PaymentData<CorrelationID, PrivateData, PublicData> {
    pub correlation_id: Option<CorrelationID>,
    pub private_data: Option<PrivateData>,
    pub public_data: Option<PublicData>,
}

#[derive(Serialize, Deserialize, JsonSchema)]
pub enum InternalPaymentState {
    Incomplete,
    Succeeded,
    Failed,
    Cancelled,
}

// constructor
async fn new<C: Context + Send>(
    ctx: &mut C,
    checkout_id: String,
    charge_amount: Money,
    args: <C as PaymentProcessor>::InitiateArgs,
) -> Result<
    Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    Error,
> {
    let mut payment = Payment {
        id: Uuid::new_v4().to_string(),
        state: PaymentState::Idle,
        checkout_id,
        charge_amount,
        data: PaymentData {
            correlation_id: None,
            private_data: None,
            public_data: None,
        },
    };

    payment.data = ctx.initiate_payment(&args).await?;
    ctx.create_payment(&payment).await?;

    Ok(payment)
}

// public methods
pub async fn queue_for_processing<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    payment: &mut Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    args: <C as PaymentProcessor>::ProcessArgs,
) -> Result<(), Error> {
    match &payment.state {
        PaymentState::Idle => {
            internal_ctx
                .payment_processing_topic
                .clone()
                .publish(&payment.id)
                .await
                .map_err(|_| {
                    new_application_error(
                        "UPSTREAM_ERROR",
                        "couldn't send message to payment processing topic",
                    )
                })?;
            payment.state = PaymentState::Processing(args);
            ctx.update_payment(payment).await
        }
        PaymentState::Processing(_) => Err(new_application_error(
            "PAYMENT_PROCESSING",
            "processing is in progress",
        )),
        PaymentState::Succeeded | PaymentState::Cancelled => Err(new_application_error(
            "PAYMENT_COMPLETED",
            "the payment has already completed",
        )),
    }
}

pub async fn process<C: Context + Send>(
    internal_ctx: &InternalContext,
    ctx: &mut C,
    payment: &mut Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    checkout: &mut Checkout,
) -> Result<(), Error> {
    match &payment.state {
        PaymentState::Processing(args) => {
            let result = ctx.process_payment(payment, args).await?;
            payment.data = result.data;
            match result.state {
                InternalPaymentState::Incomplete | InternalPaymentState::Failed => {
                    payment.state = PaymentState::Idle
                }
                InternalPaymentState::Succeeded => payment.state = PaymentState::Succeeded,
                InternalPaymentState::Cancelled => payment.state = PaymentState::Cancelled,
            }

            ctx.update_payment(payment).await?;

            match payment.state {
                PaymentState::Succeeded => {
                    internal_ctx
                        .payment_processing_topic
                        .clone()
                        .publish(&payment.checkout_id)
                        .await
                        .map_err(|_| {
                            new_application_error(
                                "UPSTREAM_ERROR",
                                "couldn't send message to payment processing topic",
                            )
                        })?;

                    checkout.complete(ctx).await
                }
                PaymentState::Cancelled => checkout.revert(ctx).await,
                _ => Ok(()),
            }
        }
        _ => Ok(()),
    }
}

pub async fn cancel<C: Context + Send>(
    ctx: &mut C,
    payment: &mut Payment<
        <C as PaymentProcessor>::CorrelationID,
        <C as PaymentProcessor>::PrivateData,
        <C as PaymentProcessor>::PublicData,
        <C as PaymentProcessor>::ProcessArgs,
    >,
    checkout: &mut Checkout,
) -> Result<(), Error> {
    match &payment.state {
        PaymentState::Cancelled => Ok(()),
        PaymentState::Idle => {
            let cancellation_result = ctx.revert(payment).await?;
            if !cancellation_result.success {
                return Err(new_application_error(
                    "CANCELLATION_UNAVAILABLE",
                    "the payment can no longer be cancelled",
                ));
            }

            payment.state = PaymentState::Cancelled;
            ctx.update_payment(payment).await?;

            checkout.revert(ctx).await
        }
        PaymentState::Processing(_) => Err(new_application_error(
            "PAYMENT_PROCESSING",
            "the payment is processing",
        )),
        PaymentState::Succeeded => Err(new_application_error(
            "PAYMENT_COMPLETED",
            "the payment has already completed",
        )),
    }
}
